package es.caib.ibkey.alfresco.ws;

/*
 * Copyright (C) 2005-2010 Alfresco Software Limited.
 *
 * This file is part of Alfresco
 *
 * Alfresco is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Alfresco is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with Alfresco. If not, see <http://www.gnu.org/licenses/>.
 */

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.rmi.RemoteException;

import javax.security.auth.callback.Callback;
import javax.security.auth.callback.CallbackHandler;
import javax.security.auth.callback.UnsupportedCallbackException;

import org.alfresco.www.ws.service.authentication._1_0.AuthenticationFault;
import org.alfresco.www.ws.service.authentication._1_0.AuthenticationResult;
import org.alfresco.www.ws.service.authentication._1_0.EndSession;
import org.alfresco.www.ws.service.authentication._1_0.StartSession;
import org.apache.axis.EngineConfiguration;
import org.apache.axis.configuration.FileProvider;
import org.apache.ws.security.WSPasswordCallback;

import es.caib.ibkey.alfresco.ws.WebServiceFactory.WebServiceFactoryConfig;


/**
 * @author Roy Wetherall
 * Modified by Pere Joseph
 */
public class AuthenticationUtils implements CallbackHandler
{
    /** WS security information */
    private static final String WS_SECURITY_INFO = 
         "<deployment xmlns='http://xml.apache.org/axis/wsdd/' xmlns:java='http://xml.apache.org/axis/wsdd/providers/java'>" +
         "   <transport name='http' pivot='java:org.apache.axis.transport.http.HTTPSender'/>" +
         "   <globalConfiguration >" +
         "     <requestFlow >" +
         "       <handler type='java:org.apache.ws.axis.security.WSDoAllSender' >" +
         "               <parameter name='action' value='UsernameToken Timestamp'/>" +
         "               <parameter name='user' value='ticket'/>" +
         "               <parameter name='passwordCallbackClass' value='es.caib.ibkey.alfresco.ws.AuthenticationUtils'/>" +
         "               <parameter name='passwordType' value='PasswordText'/>" +
         "           </handler>" +
         "       <handler name='cookieHandler' type='java:es.caib.ibkey.alfresco.ws.CookieHandler' />" +
         "     </requestFlow >" +
         "   </globalConfiguration>" +
         "</deployment>";
    
    /** Thread local containing the current authentication details */
    private static ThreadLocal authenticationDetails = new ThreadLocal();
    
    /**
     * Start a session
     * @param config 
     * 
     * @param username
     * @param password
     * @throws AuthenticationFault
     * @throws InternalErrorException 
     */
    public static void startSession(WebServiceFactoryConfig config, String username, String password)
        throws AuthenticationFault, InternalErrorException
    {
        try
        {
        	if(authenticationDetails.get()!=null) throw new InternalErrorException("Current Alfresco session already exists");
        	
            // Start the session
            AuthenticationResult result = WebServiceFactory.getAuthenticationService(config).startSession(new StartSession(username, password)).getStartSessionReturn();           
            
            // Store the ticket for use later
            authenticationDetails.set(new AuthenticationDetails(result.getUsername(), result.getTicket(), result.getSessionid()));
        }
        catch (RemoteException exception)
        {
            if (exception instanceof AuthenticationFault)
            {
                // Rethrow the authentication exception
                throw (AuthenticationFault)exception;
            }
            else
            {
                // Throw the exception as a wrapped runtime exception
                throw new InternalErrorException("Error starting session.", exception);
            }
        }             
    }
    
    /**
     * Start a session
     * @param config 
     * 
     * @param username
     * @param password
     * @param timeoutInterval timeout interval
     * @throws AuthenticationFault
     */
    public static void startSession(WebServiceFactoryConfig config, String username, String password, long timeoutInterval)
    	throws AuthenticationFault, InternalErrorException
	{
		startSession(config,username, password);
	
		AuthenticationDetails ad = getAuthenticationDetails();
		ad.setTimeoutInterval(timeoutInterval);
	}
    
    public static void setAuthenticationDetails(AuthenticationDetails authenticationDetails)
    {
    	AuthenticationUtils.authenticationDetails.set(authenticationDetails);
    }
    
    /**
	 * @return if timeoutInterval is not set return false.
	 */
    public static boolean isCurrentTicketTimedOut(WebServiceFactoryConfig config) throws InternalErrorException
    {
    	boolean to = getAuthenticationDetails().isTimedOut();
    	
    	if (to)
    		endSession(config);
    	
    	return to;
    }


    /**
     * Ends the current session
     * @param config 
     */
    public static void endSession(WebServiceFactoryConfig config) throws InternalErrorException
    {
        AuthenticationDetails authenticationDetails = (AuthenticationDetails) AuthenticationUtils.authenticationDetails.get();
        if (authenticationDetails != null)
        {
            try
            {
                WebServiceFactory.getAuthenticationService(config).endSession(new EndSession(authenticationDetails.getTicket()));
                AuthenticationUtils.authenticationDetails.remove();
            }
            catch (RemoteException exception)
            {
                exception.printStackTrace();
                throw new InternalErrorException("Error ending session.", exception);
            }
        }
    }
    
    /**
     * Get the ticket for the current authentication details on the current thread
     * 
     * @return  String  the ticket
     */
    public static String getTicket()
    {
        String result = null;
        AuthenticationDetails authDetails = (AuthenticationDetails) AuthenticationUtils.authenticationDetails.get();
        if (authDetails != null)
        {
            result = authDetails.getTicket();
        }
        return result;
    }
    
    /**
     * Get the authentication details for the current thread
     * 
     * @return  the authentication details
     */
    public static AuthenticationDetails getAuthenticationDetails()
    {
        return (AuthenticationDetails) AuthenticationUtils.authenticationDetails.get();
    }
    
    /**
     * The implementation of the passwrod call back used by the WS Security
     * 
     * @see javax.security.auth.callback.CallbackHandler#handle(javax.security.auth.callback.Callback[])
     */
    public void handle(Callback[] callbacks) throws IOException, UnsupportedCallbackException
    {
       for (int i = 0; i < callbacks.length; i++) 
       {
          if (callbacks[i] instanceof WSPasswordCallback) 
          {
             WSPasswordCallback pc = (WSPasswordCallback)callbacks[i];
             String ticket = AuthenticationUtils.getTicket();
             if (ticket == null)
             {
                 throw new IOException("Ticket could not be found when calling callback handler.");
             }
             pc.setPassword(ticket);
          }
          else 
          {
             throw new UnsupportedCallbackException(callbacks[i], "Unrecognized Callback");
          }
       }
    }
    
    /**
     * Gets the engine configuration used to create the web service references
     * 
     * @return  EngineConfiguration     the engine configuration
     */
    public static EngineConfiguration getEngineConfiguration()
    {
        return new FileProvider(new ByteArrayInputStream(WS_SECURITY_INFO.getBytes()));
    }    
}
