package es.caib.ibkey.alfresco.ws;

/*
 * Copyright (C) 2005-2010 Alfresco Software Limited.
 *
 * This file is part of Alfresco
 *
 * Alfresco is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Alfresco is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with Alfresco. If not, see <http://www.gnu.org/licenses/>.
 */

import javax.xml.rpc.ServiceException;

import org.alfresco.www.ws.service.action._1_0.ActionServiceLocator;
import org.alfresco.www.ws.service.action._1_0.ActionServiceSoapBindingStub;
import org.alfresco.www.ws.service.authentication._1_0.AuthenticationServiceLocator;
import org.alfresco.www.ws.service.authentication._1_0.AuthenticationServiceSoapBindingStub;
import org.alfresco.www.ws.service.content._1_0.ContentServiceLocator;
import org.alfresco.www.ws.service.content._1_0.ContentServiceSoapBindingStub;
import org.alfresco.www.ws.service.repository._1_0.RepositoryServiceLocator;
import org.alfresco.www.ws.service.repository._1_0.RepositoryServiceSoapBindingStub;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import es.caib.ibkey.bpm.common.IBkeyConfig;



/**
 * 
 * 
 * @author Roy Wetherall
 * Modificat per Pere Joseph
 */
public final class WebServiceFactory
{
    /** Log */
    private static Log logger = LogFactory.getLog(WebServiceFactory.class);
    
    /** Service addresses */
    private static final String AUTHENTICATION_SERVICE_ADDRESS  = "/AuthenticationService";
    private static final String REPOSITORY_SERVICE_ADDRESS      = "/RepositoryService";
    private static final String CONTENT_SERVICE_ADDRESS         = "/ContentService";
    private static final String AUTHORING_SERVICE_ADDRESS       = "/AuthoringService";
    private static final String CLASSIFICATION_SERVICE_ADDRESS  = "/ClassificationService";
    private static final String ACTION_SERVICE_ADDRESS          = "/ActionService";
    private static final String ACCESS_CONTROL_ADDRESS          = "/AccessControlService";
    private static final String ADMINISTRATION_ADDRESS          = "/AdministrationService";
    private static final String DICTIONARY_SERVICE_ADDRESS      = "/DictionaryService";
    

    /**
     * configura propietats de webservice
     * @author u91940
     *
     */
    public static class WebServiceFactoryConfig{
        private String endpointAddress=null;
        private int timeoutMilliseconds=-1;
        
        public WebServiceFactoryConfig(String endpointAddress,int timeoutMilliseconds) {
    
	    	this.endpointAddress=endpointAddress;
	    	this.timeoutMilliseconds=timeoutMilliseconds;
        }
	}
    
    
    /**
     * Get the authentication service from system properties
     * 
     * @return
     * @throws InternalErrorException 
     */
    public static AuthenticationServiceSoapBindingStub getAuthenticationService() throws InternalErrorException
    {
    	return getAuthenticationService(null);
    }

    
    /**
     * Get the authentication service
     * 
     * @return
     * @throws InternalErrorException 
     */
    public static AuthenticationServiceSoapBindingStub getAuthenticationService(WebServiceFactoryConfig config) throws InternalErrorException
    {
    	AuthenticationServiceSoapBindingStub authenticationService = null;
        try 
        {
            // Get the authentication service
            AuthenticationServiceLocator locator = new AuthenticationServiceLocator();
            if(config!= null)
            	locator.setAuthenticationServiceEndpointAddress(config.endpointAddress + AUTHENTICATION_SERVICE_ADDRESS);
            else
            	locator.setAuthenticationServiceEndpointAddress(getEndpointAddress() + AUTHENTICATION_SERVICE_ADDRESS);                
            authenticationService = (AuthenticationServiceSoapBindingStub)locator.getAuthenticationService();
        }
        catch (ServiceException jre) 
        {
        	if (logger.isDebugEnabled() == true)
            {
        		if (jre.getLinkedCause() != null)
                {
        			jre.getLinkedCause().printStackTrace();
                }
            }
   
            throw new InternalErrorException("Error creating authentication service: " + jre.getMessage(), jre);
        }        
        
        //config can be in system properties or in instance
        if(config!=null)
        	authenticationService.setTimeout(config.timeoutMilliseconds);
        else
        	authenticationService.setTimeout(getTimeoutMilliseconds());
        
        return authenticationService;
    }
    
    
    /**
     * Get the repository service from system properties
     * 
     * @return
     * @throws InternalErrorException 
     */
    public static RepositoryServiceSoapBindingStub getRepositoryService() throws InternalErrorException
    {
    	return getRepositoryService(null);
    }
    
    /**
     * Get the repository service
     * 
     * @return
     * @throws InternalErrorException 
     */
    public static RepositoryServiceSoapBindingStub getRepositoryService(WebServiceFactoryConfig config) throws InternalErrorException
    {
    	RepositoryServiceSoapBindingStub repositoryService = null;
        try 
        {
            // Get the authentication service
            RepositoryServiceLocator locator = new RepositoryServiceLocator(AuthenticationUtils.getEngineConfiguration());
            if(config!=null)
            	locator.setRepositoryServiceEndpointAddress(config.endpointAddress + REPOSITORY_SERVICE_ADDRESS);
            else
            	locator.setRepositoryServiceEndpointAddress(getEndpointAddress() + REPOSITORY_SERVICE_ADDRESS);                

            repositoryService = (RepositoryServiceSoapBindingStub)locator.getRepositoryService();
        }
        catch (ServiceException jre) 
        {
        	if (logger.isDebugEnabled() == true)
            {
        		if (jre.getLinkedCause() != null)
                {
        			jre.getLinkedCause().printStackTrace();
                }
            }
   
            throw new InternalErrorException("Error creating authentication service: " + jre.getMessage(), jre);
        }        
        
        //config can be in system properties or in instance
        if(config!=null)
        	repositoryService.setTimeout(config.timeoutMilliseconds);
        else
        	repositoryService.setTimeout(getTimeoutMilliseconds());
        
        return repositoryService;
    }
    
   

    /**
     * Get the content service from system properties
     * 
     * @return
     * @throws InternalErrorException 
     */
    public static ContentServiceSoapBindingStub getContentService() throws InternalErrorException
    {
    	return getContentService(null);
    }
    
    /**
     * Get the content service
     * 
     * @return
     * @throws InternalErrorException 
     */
    public static ContentServiceSoapBindingStub getContentService(WebServiceFactoryConfig config) throws InternalErrorException
    {
    	ContentServiceSoapBindingStub contentService = null;
        try 
        {
            // Get the authentication service
        	ContentServiceLocator locator = new ContentServiceLocator(AuthenticationUtils.getEngineConfiguration());
        	 if(config!=null)
             	locator.setContentServiceEndpointAddress(config.endpointAddress + CONTENT_SERVICE_ADDRESS);
             else
             	locator.setContentServiceEndpointAddress(getEndpointAddress() + CONTENT_SERVICE_ADDRESS);   
                
            contentService = (ContentServiceSoapBindingStub)locator.getContentService();
        }
        catch (ServiceException jre) 
        {
        	if (logger.isDebugEnabled() == true)
            {
        		if (jre.getLinkedCause() != null)
                {
        			jre.getLinkedCause().printStackTrace();
                }
            }
   
            throw new InternalErrorException("Error creating authentication service: " + jre.getMessage(), jre);
        }        
        
        // Time out after a minute
        //config can be in system properties or in instance
        if(config!=null)
        	contentService.setTimeout(config.timeoutMilliseconds);
        else
        	contentService.setTimeout(getTimeoutMilliseconds());
        
        return contentService;
    }
    
    /**
     * Get the repository service from system properties
     * 
     * @return
     * @throws InternalErrorException 
     */
    public static ActionServiceSoapBindingStub getActionService() throws InternalErrorException
    {
    	return getActionService(null);
    }
    
    /**
     * Get the repository service
     * 
     * @return
     * @throws InternalErrorException 
     */
    public static ActionServiceSoapBindingStub getActionService(WebServiceFactoryConfig config) throws InternalErrorException
    {
    	ActionServiceSoapBindingStub actionService = null;
        try 
        {
            // Get the authentication service
        	ActionServiceLocator locator = new ActionServiceLocator(AuthenticationUtils.getEngineConfiguration());
            if(config!=null)
            	locator.setActionServiceEndpointAddress(config.endpointAddress + ACTION_SERVICE_ADDRESS);
            else
            	locator.setActionServiceEndpointAddress(getEndpointAddress() + ACTION_SERVICE_ADDRESS);                

            actionService = (ActionServiceSoapBindingStub)locator.getActionService();
        }
        catch (ServiceException jre) 
        {
        	if (logger.isDebugEnabled() == true)
            {
        		if (jre.getLinkedCause() != null)
                {
        			jre.getLinkedCause().printStackTrace();
                }
            }
   
            throw new InternalErrorException("Error creating action service: " + jre.getMessage(), jre);
        }        
        
        //config can be in system properties or in instance
        if(config!=null)
        	actionService.setTimeout(config.timeoutMilliseconds);
        else
        	actionService.setTimeout(getTimeoutMilliseconds());
        
        return actionService;
    }     

    
    /**
     * Gets the configured end-point address
     */
    public static String getEndpointAddress()
    {
        return IBkeyConfig.getAlfrescoWSEndpointAddress();
    }

    private static int getTimeoutMilliseconds() {
    	return Integer.parseInt(IBkeyConfig.getAlfrescoWSTimeoutMilis());
	}

}
