package es.caib.ibkey.oppenoffice.helper;


import org.apache.log4j.Logger;

import com.sun.star.beans.UnknownPropertyException;
import com.sun.star.beans.XPropertySet;
import com.sun.star.bridge.XBridge;
import com.sun.star.bridge.XBridgeFactory;
import com.sun.star.comp.helper.Bootstrap;
import com.sun.star.comp.helper.BootstrapException;
import com.sun.star.connection.ConnectionSetupException;
import com.sun.star.connection.NoConnectException;
import com.sun.star.connection.XConnection;
import com.sun.star.connection.XConnector;
import com.sun.star.frame.TerminationVetoException;
import com.sun.star.frame.XDesktop;
import com.sun.star.frame.XTerminateListener;
import com.sun.star.lang.EventObject;
import com.sun.star.lang.WrappedTargetException;
import com.sun.star.lang.XComponent;
import com.sun.star.lang.XMultiComponentFactory;
import com.sun.star.uno.Exception;
import com.sun.star.uno.UnoRuntime;
import com.sun.star.uno.XComponentContext;


/**
 * A bootstrap connector which handles a connection to an OOo server.
 * 
 * Most of the source code in this class has been taken from the Java class
 * "Bootstrap.java" (Revision: 1.15) from the UDK projekt (Uno Software Develop-
 * ment Kit) from OpenOffice.org (http://udk.openoffice.org/). The source code
 * is available for example through a browser based online version control
 * access at http://udk.openoffice.org/source/browse/udk/. The Java class
 * "Bootstrap.java" is there available at
 * http://udk.openoffice.org/source/browse/udk/javaunohelper/com/sun/star/comp/helper/Bootstrap.java?view=markup
 * 
 * The idea to develop this BootstrapConnector comes from the blog "Getting
 * started with the OpenOffice.org API part III : starting OpenOffice.org with
 * jars not in the OOo install dir by Wouter van Reeven"
 * (http://technology.amis.nl/blog/?p=1284) and from various posts in the
 * "(Unofficial) OpenOffice.org Forum" at http://www.oooforum.org/ and the
 * "OpenOffice.org Community Forum" at http://user.services.openoffice.org/
 * complaining about "no office executable found!".
 * 
 * 
 * Modified by Pere Joseph 02-09-2009 to allow concurrency with one instance of openoffice and clean resources on disconect
 */
public class BootstrapConnector implements XTerminateListener {

    /** The OOo server. */
    private OOoServer oooServer;

    private Object desktop; 
    private XDesktop xDesktop;

    /** The connection string which has ben used to establish the connection. */
    private String oooConnectionString;

	private Boolean canConnect=new Boolean(true);
	
	
	/** Components neede to handle the connection **/
	XComponentContext xLocalContext ;
    XConnector xConnector=null;
    XConnection xConnection=null;
    XBridge xBridge=null;
	
    
    
    private Logger logger=Logger.getLogger(BootstrapConnector.class);
    
	/**
     * Constructs a bootstrap connector which uses the folder of the OOo
     * installation containing the soffice executable.
     * 
     * @param   oooExecFoder   The folder of the OOo installation containing the soffice executable
     */
    public BootstrapConnector(String oooExecFolder) {
        
        this.oooServer = new OOoServer(oooExecFolder);
        this.oooConnectionString = null;
    }

    /**
     * Constructs a bootstrap connector which connects to the specified
     * OOo server.
     * 
     * @param   oooServer   The OOo server
     */
    public BootstrapConnector(OOoServer oooServer) {

        this.oooServer = oooServer;
        this.oooConnectionString = null;
        

    }

    /**
     * Connects to an OOo server using the specified accept option and
     * connection string and returns a component context for using the
     * connection to the OOo server.
     * 
     * The accept option and the connection string should match to get a
     * connection. OOo provides to different types of connections:
     * 1) The socket connection
     * 2) The named pipe connection
     * 
     * To create a socket connection a host and port must be provided.
     * For example using the host "localhost" and the port "8100" the
     * accept option and connection string looks like this:
     * - accept option    : -accept=socket,host=localhost,port=8100;urp;
     * - connection string: uno:socket,host=localhost,port=8100;urp;StarOffice.ComponentContext
     * 
     * To create a named pipe a pipe name must be provided. For example using
     * the pipe name "oooPipe" the accept option and connection string looks
     * like this:
     * - accept option    : -accept=pipe,name=oooPipe;urp;
     * - connection string: uno:pipe,name=oooPipe;urp;StarOffice.ComponentContext
     * 
     * @param   oooAcceptOption       The accept option
     * @param   oooConnectionString   The connection string
     * @return                        The component context
     */
    public XComponentContext connect(String oooAcceptOption, String oooConnectionString) throws BootstrapException {
    	XComponentContext xContext = null;
    	
    	boolean canConnect_=true;
    	do{
	    	synchronized(canConnect){
	    		canConnect_=canConnect.booleanValue();

    			if(canConnect_==true){
						        this.oooConnectionString = oooConnectionString;
						
						        
						        try {
						            // get local context
						            xLocalContext = getLocalContext();
						            
						            //start openoffice if objective service is in localhost
					            	oooServer.start(oooAcceptOption);
					            	
					            	// wait until office is started
						            for (int i = 0;; ++i) {
						            	logger.info("Intento de conexión al openoffice");
						            	try {
						                	//get the connector
						                	
								            xConnector=getConnector(xLocalContext,oooConnectionString);
								            logger.debug("tenemos el xConnector");
								            
								            xConnection=getConnection(xConnector,oooConnectionString);
								            logger.debug("tenemos la xConnection");
								            
								            xBridge=getBridge(xLocalContext,xConnection);
								            logger.debug("tenemos el xBridge");
								            
						                    //now get the remote context
						                	xContext = getRemoteContext(xBridge);
						                	logger.debug("tenemos el xContext");
						                	
						                    // get desktop Service
						                    desktop = xContext.getServiceManager().createInstanceWithContext("com.sun.star.frame.Desktop",xContext);
						                    xDesktop = (XDesktop) UnoRuntime.queryInterface(XDesktop.class, desktop);
						                    logger.debug("tenemos el xDesktop");
						                    
						                    // add this as a terminateListener to controll concurrency
						                    xDesktop.addTerminateListener(this);
						                    logger.debug("Callback terminateListener registrado");
						                    
						                    break;
						                } catch ( Exception ex ) {
						                    // Wait 500 ms, then try to connect again, but do not wait
						                    // longer than 5 min (= 600 * 500 ms) total:
						                	try{
						                		logger.warn("Intento de conexión "+i+" al openoffice fallido: "+ex.getMessage());
						                		logger.error(ex.getMessage(),ex);
						                	
							                	if(xDesktop!=null)xDesktop.terminate();
							                    if(xBridge!=null){
							                    	XComponent xcomponent = (XComponent) UnoRuntime.queryInterface(XComponent.class, xBridge); 
							                    	if(xcomponent!=null)xcomponent.dispose();
							                    }
							                	if(xConnection!=null)xConnection.close();
							                	if(xConnector!=null){
							                		XComponent xcomponent = (XComponent) UnoRuntime.queryInterface(XComponent.class,xConnector);
							                		if(xcomponent!=null)xcomponent.dispose();
							                	}
							                	
							                }catch(Exception ex1){
							                	logger.warn("Intento de liberación de conexión al openoffice fallido: "+ex.getMessage());
							                }finally{
							                	try{
								                	Thread.sleep(500);
							                	}catch(InterruptedException ie){
							                		//lo ignoramos
							                	}
							                	
							                	if (i == 60) {
							                        throw new BootstrapException("Realizados 60 reintentos de conexión al openoffice: "+ex.toString());
							                    }
							                    
							                }
						                }
						            }

						        } catch (java.lang.Exception e) {
						        	logger.error(e.getMessage(),e) ;
						            throw new BootstrapException(e);
						        }
    			}	    	
	    	}
	    	if(canConnect_==false){
	    		String tempString=new String();
	    		synchronized(tempString){
	    			try {
						tempString.wait(500);
					} catch (InterruptedException e) {
						//ignoramos esto
					}
	    		}
	    	}
    	}
	    while(canConnect_==false);
	    
	    return xContext;
    }
    /**
     * Obtains a bridge
     * @param mxComponentContext
     * @param connection
     * @return
     * @throws Exception
     */
    private XBridge getBridge(XComponentContext mxComponentContext,XConnection connection) throws Exception {
    	Object x = mxComponentContext.getServiceManager().createInstanceWithContext("com.sun.star.bridge.BridgeFactory", mxComponentContext); 

        XBridgeFactory xBridgeFactory = (XBridgeFactory) UnoRuntime.queryInterface(XBridgeFactory.class, x); 
        if (xBridgeFactory == null) 
        	logger.fatal("Bridge factoriy is null"); 
        
        // this is the bridge that you will dispose 
        XBridge bridge = xBridgeFactory.createBridge("", "urp", connection, null); 
        return bridge;

	}

	/**
     * Obtains the connection
     * @return
     * @throws ConnectionSetupException 
     * @throws NoConnectException 
     */
    private XConnection getConnection(XConnector xConnector,String ooConnectionString) throws NoConnectException, ConnectionSetupException {
    	//ooConnectionString="pipe,name=OOPipe;urp;StarOffice.ComponentContext"
        XConnection connection = xConnector.connect(ooConnectionString); 
        if (connection == null) 
            logger.fatal("Connection is null"); 
        
        return connection;
	}

	/**
     * Gets the connector
     * @return
     * @throws Exception 
     */
    private XConnector getConnector(XComponentContext mxComponentContext,String ooConnectionString) throws Exception {
 	
    	Object x = mxComponentContext.getServiceManager().createInstanceWithContext("com.sun.star.connection.Connector", mxComponentContext); 
        
        XConnector xConnector = (XConnector) UnoRuntime.queryInterface(XConnector.class, x);
        
        return xConnector;

	}

	/**
     * Disconnects from an OOo server using the connection string from the
     * previous connect.
     * 
     * If there has been no previous connect, the disconnects does nothing.
     * 
     * If there has been a previous connect, disconnect tries to terminate
     * the OOo server and kills the OOo server process the connect started.
     */
    public void disconnect() {
    	
    	
        if (oooConnectionString == null)
            return;

        // disconnect from remote openoffice
        oooConnectionString = null;
        XComponent xcomponent = (XComponent) UnoRuntime.queryInterface(XComponent.class, xBridge); 
        
        // Closing the bridge 
        xcomponent.dispose();
        oooServer.disconnect();



    }

    /**
     * Kills the OpenOffice instance
     */
    public void kill(){
       	oooServer.kill();

    }
    /**
     * Create default local component context.
     * 
     * @return      The default local component context
     * @throws java.lang.Exception 
     */
    private XComponentContext getLocalContext() throws java.lang.Exception {

        XComponentContext xLocalContext = Bootstrap.createInitialComponentContext(null);
        if (xLocalContext == null) {
            throw new BootstrapException("no local component context!");
        }
        return xLocalContext;
    }

    /**
     * Try to connect to office.
     * 
     * @return      The remote component context
     *
     * @throws WrappedTargetException 
     * @throws UnknownPropertyException 
	*/
    private XComponentContext getRemoteContext(XBridge bridge) throws UnknownPropertyException, WrappedTargetException{
 
	    // get the remote instance 
	    Object x = bridge.getInstance("StarOffice.ServiceManager"); 
	    // Query the initial object for its main factory interface 
	    XMultiComponentFactory mxMCF = (XMultiComponentFactory) UnoRuntime.queryInterface(XMultiComponentFactory.class, x); 
	
	    // retrieve the component context (it's not yet exported from the 
	    // office) 
	    // Query for the XPropertySet interface. 
	    XPropertySet xProperySet = (XPropertySet) UnoRuntime.queryInterface(XPropertySet.class, mxMCF); 
	
	    // Get the default context from the office server. 
	    Object oDefaultContext = xProperySet.getPropertyValue("DefaultContext"); 
	
	    // Query for the interface XComponentContext. 
	    XComponentContext mxComponentContext = (XComponentContext) UnoRuntime.queryInterface(XComponentContext.class, oDefaultContext);
	    return mxComponentContext;
    }
    
    /**
     * Bootstraps a connection to an OOo server in the specified soffice
     * executable folder of the OOo installation using the specified accept
     * option and connection string and returns a component context for using
     * the connection to the OOo server.
     * 
     * The accept option and the connection string should match in connection
     * type and pipe name or host and port to get a connection.
     * 
     * @param   oooExecFolder         The folder of the OOo installation containing the soffice executable
     * @param   oooAcceptOption       The accept option
     * @param   oooConnectionString   The connection string
     * @return                        The component context
     */
    public static final XComponentContext bootstrap(String oooExecFolder, String oooAcceptOption, String oooConnectionString) throws BootstrapException {

        BootstrapConnector bootstrapConnector = new BootstrapConnector(oooExecFolder);
        return bootstrapConnector.connect(oooAcceptOption, oooConnectionString);
    }

    /**
     * Handles termination notification after it can be cancelled 
     */
	public void notifyTermination(EventObject arg0) {

	}

	/**
	 * Handles termination notification, and here is allowed to cancel the termination
	 */
	public void queryTermination(EventObject arg0) throws TerminationVetoException {
		// si estamos conectados, cancelamos el cierre
		synchronized(canConnect){
			if(oooConnectionString!=null) {
				throw new TerminationVetoException();
			}else{
				canConnect=new Boolean(false);
			}		
		}
		
	}

	public void disposing(EventObject arg0) {	
		synchronized(canConnect){
			canConnect=new Boolean(true);
		}
	}
}