package es.caib.ibkey.oppenoffice.helper;

import com.sun.star.comp.helper.BootstrapException;
import com.sun.star.lib.util.NativeLibraryLoader;

import es.caib.ibkey.utils.Log4jPrintStreamAdapter;

import java.io.BufferedReader;
import java.io.File;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.IOException;
import java.io.PrintStream;
import java.net.MalformedURLException;
import java.net.URL;
import java.net.URLClassLoader;
import java.util.ArrayList;
import java.util.List;

import org.apache.log4j.Logger;



/**
 * Starts and stops an OOo server.
 * 
 * Most of the source code in this class has been taken from the Java class
 * "Bootstrap.java" (Revision: 1.15) from the UDK projekt (Uno Software Develop-
 * ment Kit) from OpenOffice.org (http://udk.openoffice.org/). The source code
 * is available for example through a browser based online version control
 * access at http://udk.openoffice.org/source/browse/udk/. The Java class
 * "Bootstrap.java" is there available at
 * http://udk.openoffice.org/source/browse/udk/javaunohelper/com/sun/star/comp/helper/Bootstrap.java?view=markup
 */
public class OOoServer {

	public Logger log = Logger.getLogger(OOoServer.class);
	
	
    /** The OOo server process. */
    private Process oooProcess;

    /** The folder of the OOo installation containing the soffice executable. */
    private String  oooInstallFolder;

    /** The options for starting the OOo server. */
    private List    oooOptions;

	private Pipe errStreamHandler;

	private Pipe outStreamHandler;

    /**
     * Constructs an OOo server which uses the folder of the OOo installation
     * containing the soffice executable and a list of default options to start
     * OOo.
     * 
     * @param   oooInstallFolder   The folder of the OOo installation containing the soffice executable
     */
    public OOoServer(String oooInstallFolder) {

        this.oooProcess = null;
        this.oooInstallFolder = oooInstallFolder;
        this.oooOptions = getDefaultOOoOptions();
    }

    /**
     * Constructs an OOo server which uses the folder of the OOo installation
     * containing the soffice executable and a given list of options to start
     * OOo.
     * 
     * @param   oooInstallFolder   The folder of the OOo installation containing the soffice executable
     * @param   oooOptions         The list of options
     */
    public OOoServer(String oooInstallFolder, List oooOptions) {

        this.oooProcess = null;
        this.oooInstallFolder = oooInstallFolder;
        this.oooOptions = oooOptions;
    }

    /**
     * Starts an OOo server which uses the specified accept option.
     * 
     * The accept option can be used for two different types of connections:
     * 1) The socket connection
     * 2) The named pipe connection
     * 
     * To create a socket connection a host and port must be provided.
     * For example using the host "localhost" and the port "8100" the
     * accept option looks like this:
     * - accept option    : -accept=socket,host=localhost,port=8100;urp;
     * 
     * To create a named pipe a pipe name must be provided. For example using
     * the pipe name "oooPipe" the accept option looks like this:
     * - accept option    : -accept=pipe,name=oooPipe;urp;
     * 
     * @param   oooAcceptOption      The accept option
     * @throws Exception 
     */
    public void start(String oooAcceptOption) throws Exception {
    	try {
	    	
	    	//only can start OO in localhost
	    	if(oooAcceptOption.indexOf("socket")!=-1 && oooAcceptOption.indexOf("localhost")==-1) return;
	    	
	    	//checkConfig();
	    	
	        // find office executable relative to this class's class loader
	        String sOffice = System.getProperty("os.name").startsWith("Windows")? "soffice.exe": "soffice";            
	
	        URL[] oooInstallFolderURL = new URL[] {new URL("file://"+oooInstallFolder+"/program")};
	        URLClassLoader loader = new URLClassLoader(oooInstallFolderURL);
	        File fOffice = NativeLibraryLoader.getResource(loader, sOffice);
	        if (fOffice == null)
	            throw new BootstrapException("No office executable found at "+oooInstallFolderURL[0].toExternalForm());
	
	        // create call with arguments
	        int arguments = (oooOptions != null)? oooOptions.size()+1: 1;
	        if (oooAcceptOption != null)
	            arguments++;
	
	        String[] oooCommand = new String[arguments];
	        oooCommand[0] = fOffice.getPath();
	
	        for (int i = 0; i < oooOptions.size(); i++) {
	            oooCommand[i+1] = (String) oooOptions.get(i);
	        }
	
	        if (oooAcceptOption != null)
	            oooCommand[arguments-1] = oooAcceptOption;
	
	        // start office process
	        oooProcess = Runtime.getRuntime().exec(oooCommand);
	       /** synchronized(oooProcess){
	        	try{
	        		oooProcess.wait(5000);
	        	}catch(InterruptedException timEx){};
	        }**/
	        outStreamHandler=new Pipe();
	        outStreamHandler.pipe(oooProcess.getInputStream(), new Log4jPrintStreamAdapter(log, Log4jPrintStreamAdapter.DEBUG), "CO> ");
	        errStreamHandler=new Pipe();
	        errStreamHandler.pipe(oooProcess.getErrorStream(), new Log4jPrintStreamAdapter(log, Log4jPrintStreamAdapter.ERROR), "CE> ");
	        
    	}catch(Exception e){
    		log.error(e.getMessage(),e) ;
    		throw e;
    	}
    }

    private void checkConfig() throws Exception{ 
		File pdfFilterResourceFile= (System.getProperty("os.name").startsWith("Windows"))
				?new File(oooInstallFolder+"/Basis/program/resource/pdffilteres.res")
				:new File(oooInstallFolder+"/basis-link/program/resource/pdffilteres.res");
		if(pdfFilterResourceFile.exists()){
			if(!pdfFilterResourceFile.delete()){
				throw new IOException("No se puede eliminar el fichero de configuración "+oooInstallFolder+"/Basis/program/resource/pdffilteres.res\nEsto puede provocar que se bloquee el hilo en caso que aparezca una notificación visual de alerta en el proceso de exportación a pdf/A.");
			}
		}
		
	}

	/**
     * Kills the OOo server process from the previous start.
     * 
     * If there has been no previous start of the OOo server, the kill does
     * nothing.
     * 
     * If there has been a previous start, kill destroys the process.
     */
    public void kill() {

        if (oooProcess != null)
        {
            oooProcess.destroy();
            oooProcess = null;
        }
    }



    /**
     * Returns the list of default options.
     * 
     * @return     The list of default options
     */
    public static List getDefaultOOoOptions() {

        ArrayList options = new ArrayList();

        options.add("-nologo");
        options.add("-nodefault");
        options.add("-norestore");
        options.add("-nocrashreport");
        options.add("-nolockcheck");
        options.add("-headless");
        options.add("-nofirststartwizard");
        return options;
    }

	public void disconnect() {
		outStreamHandler.unpipe();
		errStreamHandler.unpipe();
	}
    
}