package es.caib.ibkey.oppenoffice.helper;


import java.util.Iterator;
import java.util.Map;
import java.util.Random;
import java.util.Vector;
import org.apache.log4j.Logger;
import com.sun.star.beans.PropertyValue;
import com.sun.star.comp.helper.BootstrapException;
import com.sun.star.container.XNameContainer;
import com.sun.star.frame.XComponentLoader;
import com.sun.star.frame.XStorable;
import com.sun.star.lang.XComponent;
import com.sun.star.lang.XMultiServiceFactory;
import com.sun.star.task.ErrorCodeIOException;
import com.sun.star.uno.Exception;
import com.sun.star.uno.UnoRuntime;
import com.sun.star.uno.XComponentContext;
import com.sun.star.util.XCloseable;

import es.caib.ibkey.bpm.conversion.exception.ConversionException;


/**
 * Classe que facilita la conversió de fitxers entre diferents formats mitjançant l'OpenOffice.
 * 
 * Està pensada per a fer servir amb una instància de l'openoffice en local, i treballant en multitasca.
 * Si es fa una conversió a PDF, per defecte s'estableix la conversió a PDF/A.
 * 
 * Per a la seva utilització amb instàncies remotes de l'openoffice, s'hauria de mirar si així funciona, o s'hauria d'implementar un mecanisme de transferència remota dels fitxers, via remoteXInputStream/remoteXOutputStream, o d'un altre manera.
 * @author u91940 Pere Joseph Rodriguez
 *
 */


public class OpenOfficeDocumentConversionHelper {
	

	/**Logger definit per a la classe **/
	public Logger log = Logger.getLogger(OpenOfficeDocumentConversionHelper.class);
	
	/**Connector amb l'openOffice **/
	public BootstrapConnector connector=null;

	/**host al que es troba la instància d'openoffice que es vol fer servir **/
	public  String host=System.getProperty("openoffice.host");
	/**port al que està escoltant la instància de l'openoffice **/
	public  String port=System.getProperty("openoffice.port");
	/** pipe amb la que es pot comunicar amb l'openoffice **/
	public  String pipe=System.getProperty("openoffice.pipe");

	/**Directori d'instal.lació local de l'openoffice. Necessari en cas que la instància destí de l'openoffice sigui localhost **/
	public  String openofficeInstallDir=System.getProperty("openoffice.folder");	
	
	
	/** Funció que converteix un document a un format especificat, i el desa al fitxer especificat.
	 * 
     * @param stringInputDocument Ruta completa del fitxer al servidor per a ser convertit 
     * @param stringConvertType Tipus al que es convertirà.
	 * @param stringOutputDocument Ruta completa del fitxer al servidor per a ser desat
	 * @param filterOptions Opcions del filtre de conversió que s'hagi d'utilitzar.
     * @return Es retorna la ruta completa del document convertit, al servidor.
     * @see stringWorkingDirectory
     */  
    public String convertDocument(String stringInputDocument, String stringConvertType,  String stringOutputDocument, Map filterOptions) 
        throws Exception
    {
    	log.info("Inici de conversió de format de document.");
        XComponentLoader xcomponentloader = null;
        XComponentContext xContext = null;
        
        String stringConvertedFile = "";
        
        try{   

            // get the remote office component context
        	connector=new BootstrapConnector(openofficeInstallDir);
        	if(host!=null){
        		xContext = connector.connect("-accept=socket,host="+host+",port="+port+";urp;", "socket,host="+host+",port="+port+"");
        	}else{
        		if(pipe==null) pipe="OOPipe_"+new Random().nextInt();
        		xContext = connector.connect("-accept=pipe,name="+pipe+";urp;", "pipe,name="+pipe+";urp;StarOffice.ComponentContext");
        	}
            // get the remote office service manager
            com.sun.star.lang.XMultiComponentFactory xMCF = xContext.getServiceManager();
            
            Object oDesktop = xMCF.createInstanceWithContext("com.sun.star.frame.Desktop", xContext);
        
            xcomponentloader = (com.sun.star.frame.XComponentLoader)
                UnoRuntime.queryInterface(com.sun.star.frame.XComponentLoader.class, oDesktop);
            
            // Preparing properties for loading the document
            PropertyValue componentLoaderPropertyvalue[] = new PropertyValue[ 1 ];
            // Setting the flag for hidding the open document
            componentLoaderPropertyvalue[ 0 ] = new PropertyValue();
            componentLoaderPropertyvalue[ 0 ].Name = "Hidden";
            componentLoaderPropertyvalue[ 0 ].Value = new Boolean(true);
            
            // Loading the wanted document
            Object objectDocumentToStore=null;
            String stringUrl=null;
            
        	stringUrl=(!stringInputDocument.startsWith("file:///"))? "file:///" + stringInputDocument:stringInputDocument;
     	
        	stringOutputDocument=(!stringOutputDocument.startsWith("file:///"))? "file:///" +stringOutputDocument:stringOutputDocument;
        	new java.net.URL(stringUrl).toString();//new java.io.File(new java.net.URL(stringUrl).toURI()).isFile()
        	objectDocumentToStore = xcomponentloader.loadComponentFromURL(
        			stringUrl.replaceAll("\\\\","/"), "_blank", 0, componentLoaderPropertyvalue ); 
            	

            
            // Getting an object that will offer a simple way to store a document to a URL.
            XStorable xstorable =
                ( XStorable ) UnoRuntime.queryInterface( XStorable.class,
                                                         objectDocumentToStore );
            
            if(xstorable==null)
            	throw new ConversionException("El format del document d'entrada no és vàlid per a realitzar el canvi de format.");
          
            Vector filterProperties=new Vector();
            // Preparing properties for converting the document
            // Setting the flag for overwriting
            PropertyValue fprop= new PropertyValue();
            fprop.Name = "Overwrite";
            fprop.Value = new Boolean(true);
            filterProperties.add(fprop);
            // Setting the filter name
            fprop= new PropertyValue();
            fprop.Name = "FilterName";
            fprop.Value = stringConvertType;
            filterProperties.add(fprop);

           
            //Setting Filter configuration           
            Vector filterData=new Vector();
    		//defaults PDF/A for pdf conversions
            if(filterOptions==null || (stringConvertType.toLowerCase().contains("pdf") && filterOptions.get("SelectPdfVersion")==null)){
            	PropertyValue fdata=new PropertyValue();	
            	fdata.Name = "SelectPdfVersion";
            	fdata.Value = new Integer(1);
            	filterData.add(fdata);
	            
            }
            
            //sets filter options received via argument
            if(filterOptions!=null){
	            Iterator it=filterOptions.keySet().iterator();
	            while(it.hasNext()){
	            	String propertyName=(String)it.next();
	            	if(propertyName.contains("OO.")){ //agafem no més les opcions d'openOffice
		            	PropertyValue fdata=new PropertyValue();	
		            	fdata.Name = propertyName.replace("OO.", ""); //eliminem l'indicador de propietat de l'openofice 
		            	fdata.Value = filterOptions.get(propertyName);
		            	filterData.add(fdata);
	            	}
	            	
	            }
            }
            
            if(filterData.size()!=0){
	            fprop= new PropertyValue();
	            fprop.Name = "FilterData";
	            fprop.Value = (PropertyValue[])filterData.toArray(new PropertyValue[filterData.size()]);
	            filterProperties.add(fprop);
            }

            // Storing and converting the document
            stringOutputDocument=stringOutputDocument.replaceAll("\\\\","/");
            xstorable.storeToURL( stringOutputDocument, (PropertyValue[])filterProperties.toArray(new PropertyValue[filterProperties.size()]) );
            
            XCloseable xcloseable = (XCloseable)UnoRuntime.queryInterface( XCloseable.class,xstorable );
            
            // Closing the converted document
            if ( xcloseable != null )
                xcloseable.close(false);
            else {
                // If Xcloseable is not supported (older versions,
                // use dispose() for closing the document
                XComponent xComponent = ( XComponent ) UnoRuntime.queryInterface(
                    XComponent.class, xstorable );
                xComponent.dispose();
            }
            

	        log.info("Conversió de format de document realitzada amb èxit.");
        // Returning the name of the converted file
    	}catch(Throwable th){
    		Logger.getLogger(OpenOfficeDocumentConversionHelper.class).error(th.getMessage(),th);
			String message=th.getMessage();
			if(th instanceof ErrorCodeIOException)
				message+=": "+((ErrorCodeIOException)th).ErrCode;
    		Exception e1=new Exception(message);
			e1.setStackTrace(th.getStackTrace());
			log.error(e1.getMessage(),e1);
			throw e1;
    	}finally{
    	 	try{
    	 		if(connector!=null) connector.disconnect();
    	 	}catch(Throwable t){}
    	}
        return stringOutputDocument;
    }
    
    /**
     * Converteix documents, o mostra el llistat de filtres de conversió disponibles
     * 
     * @param args
     * @throws Exception
     */

    public static void main(String [] args) throws Exception{
    	
    	OpenOfficeDocumentConversionHelper handler=new OpenOfficeDocumentConversionHelper();
    	if(args.length==3){
    		try {
				handler.convertDocument(args[0], args [1], args[2],null);
			} catch (Exception e) {
				
				Logger.getLogger(OpenOfficeDocumentConversionHelper.class).error(e.getMessage(),e);
			}
    	}else{
    		
            // get the remote office component context
            XComponentContext xContext=null;
			try {
	        	handler.connector=new BootstrapConnector(handler.openofficeInstallDir);
				xContext = handler.connector.connect("-accept=socket,host="+handler.host+",port="+handler.port+";urp;", "socket,host="+handler.host+",port="+handler.port+"");

			} catch (BootstrapException e) {
				Logger.getLogger(OpenOfficeDocumentConversionHelper.class).error(e.getMessage(),e);
				System.exit(1);
			}
			handler.log.info("Connected to a running office ...");

            // get the remote office service manager
            com.sun.star.lang.XMultiComponentFactory xMCF = xContext.getServiceManager();
            
            XMultiServiceFactory mServiceFactory   = (XMultiServiceFactory)UnoRuntime.queryInterface( 
            XMultiServiceFactory.class, xMCF); 
                        
            Object object=null;
			try {
				object = mServiceFactory.createInstance("com.sun.star.document.FilterFactory");
			} catch (Exception e) {
				handler.log.error(e.getMessage(),e);
				System.exit(1);
			} 
			
			XNameContainer ff = (XNameContainer)UnoRuntime.queryInterface(com.sun.star.container.XNameContainer.class,object);
			String types[]=ff.getElementNames();
            
    		String help="Usage: ConversionHandler inputFile outputFormat outputFileExtension \n"+
    		"\n" +
    		"Configuration of each filter is defined in Common.xcs\n\n" +
    		"Possible output format filter values are : \n";
    		StringBuffer buf=new StringBuffer(help);
    		for (int i=0;i<types.length;i++){
    			buf.append(types[i]);
    			buf.append("\n");
    		}
    		
    		System.out.println(buf.toString());
    		
    		if(handler.connector!=null) handler.connector.disconnect();
    	}
    }
}
