/**
 * 
 */
package es.caib.ibkey.bpm.pdf.handler;


import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.util.Iterator;
import java.util.Vector;

import org.apache.log4j.Logger;
import org.jbpm.graph.exe.ExecutionContext;

import com.lowagie.text.DocumentException;
import com.lowagie.text.Image;
import com.lowagie.text.Rectangle;
import com.lowagie.text.pdf.BarcodePDF417;
import com.lowagie.text.pdf.BaseFont;
import com.lowagie.text.pdf.PdfContentByte;
import com.lowagie.text.pdf.PdfReader;
import com.lowagie.text.pdf.PdfStamper;

import es.caib.ibkey.bpm.common.SyntaxQueryResolver;
import es.caib.ibkey.bpm.common.handler.AbstractIndexedHandler;
import es.caib.ibkey.bpm.document.stage.URLDescription;
import es.caib.ibkey.bpm.document.stage.impl.StageInfoEstampatPDF;

/**
 * Handler que permet funcionalitats d'estampat d'un PDF.
 * Configuració del Handler:
 * 
 *  stampPhrase: indica la frase a estampar, que puede ser una cadena, o si coincide con la descripción de una de las URLs asociadas al workflow, devolverá esta URL.
 * @author u91940
 *
 */
public class PDFStamperHandler extends AbstractIndexedHandler {

	public int SIGNATURE_HORIZONTAL_SPLIT_LEN=65;
	public int SIGNATURE_VERTICAL_SPLIT_LEN=100;
	public int SIGNATURE_SPLIT_RANGE=10;
	public int SIGNATURE_SPLIT_MARGIN=5;


	protected byte [] input=null;
	protected int stampPosition=0;
	protected String stampPhrase=null;
	private ByteArrayInputStream stampedStream;

	private int translateParams(String stampPositionString){

		if("TOP".equals(stampPositionString)) 	return PDF_STAMP_POSITION_TOP;
		if("BOTOM".equals(stampPositionString)) return PDF_STAMP_POSITION_BOTTOM;
		if("LEFT".equals(stampPositionString)) 	return PDF_STAMP_POSITION_LEFT;
		if("RIGHT".equals(stampPositionString)) return PDF_STAMP_POSITION_RIGHT;
		if("NONE".equals(stampPositionString)) 	return PDF_STAMP_POSITION_NONE;	
		return PDF_STAMP_POSITION_NONE;
	}
	
	protected void processParameters(ExecutionContext ctx){
		
		StageInfoEstampatPDF stgInfo=(StageInfoEstampatPDF)getStageActual();
				
		input=(byte [])getInputObject();
		stampPosition=translateParams(stgInfo.getStampPosition());
		stampPhrase=SyntaxQueryResolver.resolve(stgInfo.getStampPhrase(),ctx.getContextInstance().getVariables());

		if(stampPhrase!=null && ! "".equals(stampPhrase)){
			Iterator kit=getStgMgr().getDownloadURL().keySet().iterator();
			while(kit.hasNext()){
				URLDescription urlDesc=(URLDescription)getStgMgr().getDownloadURL().get((String)kit.next());
				if(stampPhrase.equals(urlDesc.getDescription())){
					stampPhrase=urlDesc.getURL();
				}
			}
		}
	}
	
	protected final void handleExecute(ExecutionContext ctx) throws Exception {
		processParameters(ctx);
		
		byte [] output=estampadoPDF417(input,stampPhrase,stampPosition);
		stampedStream=new ByteArrayInputStream(output);
		setOutputObject(stampedStream);
		setOutputContentType("application/pdf");
	}


	protected void handleRollBack(ExecutionContext ctx) {
		// no hem de fer res

	}


	
	private byte [] estampadoPDF417(byte[] pdfInput, String line, int stampPosition) throws Exception {

		
		PdfReader reader=null;
		PdfStamper stamp=null;
		try {
			reader = new PdfReader(pdfInput);
			int numberOfPages = reader.getNumberOfPages();

			BaseFont bf = BaseFont.createFont(BaseFont.HELVETICA, BaseFont.WINANSI, BaseFont.EMBEDDED);
			
			ByteArrayOutputStream stampedDoc = new ByteArrayOutputStream();
			stamp= new PdfStamper(reader, stampedDoc);

			if(stampPosition!=PDF_STAMP_POSITION_NONE){
	
				float pdf417_X = 20;
				float pdf417_Y = 20;
				int textAlign = PdfContentByte.ALIGN_LEFT;
				float text_X = 30;
				float text_Y = 150;
				float rotation = 90;
		
				
				BarcodePDF417 pdf417 = new BarcodePDF417();
				pdf417.setText(line);
				Image image = pdf417.getImage();
				
				for (int i = 1; i <=numberOfPages; i++) {
					PdfContentByte over = stamp.getOverContent(i);
		
					Rectangle pageSize = reader.getPageSize(i);
					float width = pageSize.getWidth();
					float height = pageSize.getHeight();
					
					int SIGNATURE_SPLIT_LEN=0;
					if(height>width)
						SIGNATURE_SPLIT_LEN=(stampPosition==PDF_STAMP_POSITION_LEFT || stampPosition==PDF_STAMP_POSITION_RIGHT)?SIGNATURE_VERTICAL_SPLIT_LEN:SIGNATURE_HORIZONTAL_SPLIT_LEN;
					else
						SIGNATURE_SPLIT_LEN=(stampPosition==PDF_STAMP_POSITION_LEFT || stampPosition==PDF_STAMP_POSITION_RIGHT)?SIGNATURE_HORIZONTAL_SPLIT_LEN:SIGNATURE_VERTICAL_SPLIT_LEN;
		
					Vector lines=new Vector();
					int counter=0;
					
					while(counter!=-1){
						if(counter+SIGNATURE_SPLIT_LEN<line.length()){
							int prev=line.substring(counter,counter+SIGNATURE_SPLIT_LEN).lastIndexOf(" "); //$NON-NLS-1$
							String next_string=line.substring(counter+SIGNATURE_SPLIT_LEN,line.length());
							int next=next_string.indexOf(" "); //$NON-NLS-1$
							//prev [-1,SPLIT_LEN]
							//next[-1,inf]
		
							int splitPos=prev+1; 
							//splitpos [-1,SPLIT_LEN+1]
		
							if(next<SIGNATURE_SPLIT_RANGE && next!=-1){
								splitPos=SIGNATURE_SPLIT_LEN+next;
								//next[0,inf]
								//splitpos [0,SPLIT_LEN+SPLIT_RANGE]
								
							}else if(prev==-1 && next>SIGNATURE_SPLIT_RANGE ){
								splitPos=SIGNATURE_SPLIT_LEN;
								//prev [0,SPLIT_LEN]
								//splitpos [0,SPLIT_LEN+SPLIT_RANGE]
							}else if(prev==-1 && next==-1){
								splitPos=SIGNATURE_SPLIT_LEN;
							}
							
							lines.add(line.substring(counter,counter+splitPos));
							counter+=splitPos;
							
						}else{
							lines.add(line.substring(counter,line.length()));
							counter=-1;
						}
					}
		
					
					switch (stampPosition) {
						case (PDF_STAMP_POSITION_LEFT):
							pdf417_X = 20;
							pdf417_Y = 20;
							textAlign = PdfContentByte.ALIGN_LEFT;
							text_X = 30;
							text_Y = pdf417_Y+image.getWidth()+20;
							rotation = 90;
							break;
						case (PDF_STAMP_POSITION_RIGHT):
							pdf417_X = width-40;
							pdf417_Y = 20;
							textAlign = PdfContentByte.ALIGN_LEFT;
							text_X = width-20;
							text_Y = pdf417_Y+image.getWidth()+20;
							rotation = 90;
							break;
						case (PDF_STAMP_POSITION_TOP):
							pdf417_X = 20;
							pdf417_Y = height-50;
							textAlign = PdfContentByte.ALIGN_LEFT;
							text_X = pdf417_X+image.getWidth()+20;
							text_Y = height-50;
							rotation = 0;
							break;
						case (PDF_STAMP_POSITION_BOTTOM):
							pdf417_X = 20;
							pdf417_Y = 30;
							textAlign = PdfContentByte.ALIGN_LEFT;
							text_X = pdf417_X+image.getWidth()+20;
							text_Y = 30;
							rotation = 0;
							break;
						default: 
					
					}
		
					image.setAbsolutePosition(pdf417_X, pdf417_Y);
					image.setRotationDegrees(rotation);
					over.addImage(image);
		
					counter=0;
					
					//Descentramos en función del número de líneas de la firma
					text_X=((stampPosition==PDF_STAMP_POSITION_LEFT || stampPosition==PDF_STAMP_POSITION_RIGHT)?text_X-(lines.size()/2)*(10+SIGNATURE_SPLIT_MARGIN):text_X);
					text_Y=(stampPosition==PDF_STAMP_POSITION_TOP || stampPosition==PDF_STAMP_POSITION_BOTTOM)?text_Y+(lines.size()/2)*(10+SIGNATURE_SPLIT_MARGIN):text_Y;
					
					//Imprimimos las líneas de la firma
					for(counter=0;counter<lines.size();counter++){
				
						over.beginText();
						over.setFontAndSize(bf, 10);
						float x= ((stampPosition==PDF_STAMP_POSITION_LEFT || stampPosition==PDF_STAMP_POSITION_RIGHT)?text_X+counter*(10+SIGNATURE_SPLIT_MARGIN):text_X);
						float y=(stampPosition==PDF_STAMP_POSITION_TOP || stampPosition==PDF_STAMP_POSITION_BOTTOM)?text_Y-counter*(10+SIGNATURE_SPLIT_MARGIN):text_Y;
						over.showTextAligned(textAlign, (String)lines.get(counter),x ,y ,rotation);
						over.endText();
		
					}
				}
			}
			if(stamp!=null){
				try{
					stamp.close();
				}catch(Exception e){
					error(e.getMessage(),e);
					throw e;
				}finally{
					if(reader!=null){
						try{
							reader.close();
						}catch(Exception e){
							error(e.getMessage(),e);
							throw e;
						}
					}

				}
			}
			byte [] resultBytes=stampedDoc.toByteArray();
			return resultBytes;
			
			
		} catch (DocumentException e) {
			error(e.getMessage(),e);
			throw e;
		}finally{
			
		}
		
	}

	/**
	 * @return the sIGNATURE_HORIZONTAL_SPLIT_LEN
	 */
	public synchronized int getSIGNATURE_HORIZONTAL_SPLIT_LEN() {
		return SIGNATURE_HORIZONTAL_SPLIT_LEN;
	}

	/**
	 * @param sIGNATUREHORIZONTALSPLITLEN the sIGNATURE_HORIZONTAL_SPLIT_LEN to set
	 */
	public synchronized void setSIGNATURE_HORIZONTAL_SPLIT_LEN(int sIGNATUREHORIZONTALSPLITLEN) {
		SIGNATURE_HORIZONTAL_SPLIT_LEN = sIGNATUREHORIZONTALSPLITLEN;
	}

	/**
	 * @return the sIGNATURE_VERTICAL_SPLIT_LEN
	 */
	public synchronized int getSIGNATURE_VERTICAL_SPLIT_LEN() {
		return SIGNATURE_VERTICAL_SPLIT_LEN;
	}

	/**
	 * @param sIGNATUREVERTICALSPLITLEN the sIGNATURE_VERTICAL_SPLIT_LEN to set
	 */
	public synchronized void setSIGNATURE_VERTICAL_SPLIT_LEN(int sIGNATUREVERTICALSPLITLEN) {
		SIGNATURE_VERTICAL_SPLIT_LEN = sIGNATUREVERTICALSPLITLEN;
	}

	/**
	 * @return the sIGNATURE_SPLIT_RANGE
	 */
	public synchronized int getSIGNATURE_SPLIT_RANGE() {
		return SIGNATURE_SPLIT_RANGE;
	}

	/**
	 * @param sIGNATURESPLITRANGE the sIGNATURE_SPLIT_RANGE to set
	 */
	public synchronized void setSIGNATURE_SPLIT_RANGE(int sIGNATURESPLITRANGE) {
		SIGNATURE_SPLIT_RANGE = sIGNATURESPLITRANGE;
	}

	/**
	 * @return the sIGNATURE_SPLIT_MARGIN
	 */
	public synchronized int getSIGNATURE_SPLIT_MARGIN() {
		return SIGNATURE_SPLIT_MARGIN;
	}

	/**
	 * @param sIGNATURESPLITMARGIN the sIGNATURE_SPLIT_MARGIN to set
	 */
	public synchronized void setSIGNATURE_SPLIT_MARGIN(int sIGNATURESPLITMARGIN) {
		SIGNATURE_SPLIT_MARGIN = sIGNATURESPLITMARGIN;
	}


	
	/**
	 * 
	 */
	private static final long serialVersionUID = 1L;

	/**
	 * Indica que la firma se añadirá en la parte superior del PDF.
	 */
	private static final int PDF_STAMP_POSITION_TOP = 1;
	
	/**
	 * Indica que la firma se añadirá en la parte inferior del PDF.
	 */
	private static final int PDF_STAMP_POSITION_BOTTOM = 2;
	
	/**
	 * Indica que la firma se añadirá en el lado izquierdo del PDF.
	 */
	private static final int PDF_STAMP_POSITION_LEFT = 4;
	
	/**
	 * Indica que la firma se añadirá en el lado derecho del PDF.
	 */
	private static final int PDF_STAMP_POSITION_RIGHT = 8;

	/**
	 * Indica que no se añadirá ni código de barras ni información del firmante.
	 */
	private static final int PDF_STAMP_POSITION_NONE = 0;

	protected void freeResources(ExecutionContext ctx) {
		if(stampedStream!=null)
			try {
				stampedStream.close();
			} catch (IOException e) {
			}
		
	}	
	
}
