package es.caib.signatura.provider.impl.common;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.security.DigestInputStream;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.security.NoSuchProviderException;

import org.bouncycastle.cms.CMSException;
import org.bouncycastle.cms.CMSProcessable;
import org.bouncycastle.cms.CMSSignedData;
import org.bouncycastle.cms.CMSSignedDataGenerator;

import es.caib.signatura.impl.SMIMEPkcs7InputStream;

import es.caib.signatura.api.Signature;
import es.caib.signatura.impl.RawSignature;
import es.caib.signatura.impl.SMIMEGenerator;
import es.caib.signatura.impl.SMIMEInputStream;

/**
 * Implementación del generador S/MIME.
 * 
 * @author 3digits
 *
 */
public class CopyOfSMIMEGeneratorImpl implements SMIMEGenerator {
	
	public CopyOfSMIMEGeneratorImpl() {}
	
	/**
	 * A partir del documento y de la firma generada sobre el mismo, se devuelve un documento en formato S/MIME a través de un InputStream.
	 * 
	 * @param document Documento que ha sido firmado.
	 * @param signature Firma del documento
	 * 
	 * @return Documento SMIME.
	 * 
	 * @throws IOException En caso de no encontrar el documento pasado por parámetro.
	 */
	public InputStream generateSMIME(InputStream document, Signature signature) throws IOException {
		InputStream in;
		if( signature instanceof RawSignature ) {
			byte [] pkcs7 = encapsulateData(signature.getPkcs7(), document);
			in = new SMIMEPkcs7InputStream(pkcs7);
		}
		else {
			in = new SMIMEInputStream (signature, document);
		}
		
		return in;
	}
	
	/**
	 * Devuelve la firma pkcs7 con los datos encapsulados dentro de la firma. Si el array de bytes <code>pkcs7</code> no és una firma
	 * bien formada el método devolverá null. No se lanzan algunas excepciones porque el método es privado
	 * y se supone que la firma que se pasa por parámetro es correcta.
	 * 
	 * @param pkcs7 array de bytes que contiene la firma pkcs7 dettached.
	 * @param data InputStream de los datos que se quieren encapsular dentro de la firma.
	 * @return firma en formato pkcs7 attached. 
	 * 
	 * @throws IOException
	 */
	private static byte[] encapsulateData(byte[] pkcs7, InputStream data) throws IOException {
		try {
			CMSSignedData cmsSignedData = new CMSSignedData(pkcs7);
			CMSSignedDataGenerator cmsSignedDataGenerator = new CMSSignedDataGenerator();
			cmsSignedDataGenerator.addSigners(cmsSignedData.getSignerInfos());
			cmsSignedDataGenerator.addCertificatesAndCRLs(cmsSignedData.getCertificatesAndCRLs("Collection","BC"));
			
			
			CMSProcessable cmsProcessable = new CopyOfSMIMEGeneratorImpl().new ProcessableInputStream(data);
			
			cmsSignedData = cmsSignedDataGenerator.generate(cmsProcessable, true, "BC");
			
			return cmsSignedData.getEncoded();
			
		}
		catch (Exception e) {
			e.printStackTrace();
		    return null;
		}
	}
	
	class ProcessableInputStream implements CMSProcessable 
    {
      private DigestInputStream in;
      MessageDigest digester;
      byte digestResult [];

      public void write(OutputStream out) throws IOException, CMSException {
        byte b[] = new byte[8192];
        int read = in.read(b);
        while (read > 0)
        {
          out.write(b, 0, read);
          read = in.read(b);
        }
        out.close();
        in.close();
        digestResult = digester.digest();
      }

      public Object getContent() {
        return in;
      }

      public ProcessableInputStream(InputStream datain) throws NoSuchAlgorithmException, NoSuchProviderException {
        super();
        digester  = MessageDigest.getInstance("SHA-1", "BC");
        in = new DigestInputStream (datain, digester);
        digestResult = null;
      }

      public byte [] getDigest() {
        return digestResult;
      }
      
    }
	
}
