package es.caib.signatura.provider.impl.common;

import es.caib.signatura.api.Signature;
import es.caib.signatura.impl.GeneradorSMIMEParalelo;

import es.caib.signatura.impl.MIMEInputStream;

import es.caib.signatura.impl.SMIMEInputStream;

import java.io.IOException;
import java.io.InputStream;

import java.io.OutputStream;

import java.security.DigestInputStream;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.security.NoSuchProviderException;
import java.security.cert.CertStore;
import java.security.cert.CertStoreParameters;
import java.security.cert.CollectionCertStoreParameters;

import java.util.ArrayList;
import java.util.Properties;

import javax.mail.Session;

import org.bouncycastle.cms.CMSException;
import org.bouncycastle.cms.CMSProcessable;
import org.bouncycastle.cms.CMSSignedData;
import org.bouncycastle.cms.CMSSignedDataGenerator;
import org.bouncycastle.cms.SignerInformationStore;

/**
 * Implementación del generador de S/MIME's con N firmas en paralelo. 
 * 
 * @author Fernando Guardiola Ruiz
 */
public class GeneradorSMIMEParaleloImpl implements GeneradorSMIMEParalelo {

  public GeneradorSMIMEParaleloImpl() {
  }
  
  /**
   * Genera un S/MIME multipart/signed PKCS-7 a partir del flujo de datos <code>contentStream</code> y el array 
   * <code>signatures</code> que contiene diversas firmas del mensaje. 
   * 
   * @param contentStream Flujo de datos con el mensaje original
   * @param signatures Array con las firmas con las que se generará el PCKS-7
   * @return Un stream del que leer el S/MIME
   * @throws Exception Si se produce algun error generando el S/MIME. 
   */
  public SMIMEInputStream generarSMIMEParalelo(InputStream contentStream, Signature[] signatures) throws Exception {
    
    Properties props = System.getProperties();
    Session session = Session.getDefaultInstance(props, null);
    String contentType = "";
    
    ArrayList listSigners = new ArrayList();
    ArrayList listCertsAndCrl = new ArrayList();
    
    //Obtenemos el contentType
    contentType  = signatures[0].getContentType();
    
    //Obtenemos dos listados, uno que contiene todos los signers de todas las firmas y el otro con los 
    //certificados y crl's.
    for (int is=0;is<signatures.length;is++) {
      Signature signature = signatures[is];                  
      CMSSignedData cmsSignedData = new CMSSignedData(signature.getPkcs7());
      SignerInformationStore signerInformationStore = cmsSignedData.getSignerInfos();
      listSigners.addAll(signerInformationStore.getSigners());
      CertStore certStore = cmsSignedData.getCertificatesAndCRLs("Collection","BC");
      CollectionCertStoreParameters collectionCertStoreParams = (CollectionCertStoreParameters)certStore.getCertStoreParameters();
      listCertsAndCrl.addAll(collectionCertStoreParams.getCollection());
    }
    
    //Creamos la signerInformationStore de la firma en paralelo y la certStore con los certificados y crl's 
    //obtenidos 
    SignerInformationStore signerInformationStoreParalell = new SignerInformationStore(listSigners);
    
    CollectionCertStoreParameters collectionCertStoreParamsParalell = new CollectionCertStoreParameters(listCertsAndCrl);    
    CertStore certStoreParallel = CertStore.getInstance("Collection",collectionCertStoreParamsParalell);
    
    //Iniciamos el generador de firmas con las store de las firma en paralelo.
    CMSSignedDataGenerator cmsSignedDataGenerator = new CMSSignedDataGenerator();
    cmsSignedDataGenerator.addSigners(signerInformationStoreParalell);
    cmsSignedDataGenerator.addCertificatesAndCRLs(certStoreParallel);
    
    //Generamos el mensaje que se firma, que es el el documento original como un mensaje MIME
    MIMEInputStream mimeInputStream = new MIMEInputStream(contentStream,contentType);
    CMSProcessable cmsProcessable = new ProcessableInputStream(mimeInputStream);      
    
    //Generamos la firma
    CMSSignedData cmsSignedDataParallel = cmsSignedDataGenerator.generate(cmsProcessable,"BC");     
    
    return new SMIMEInputStream(cmsSignedDataParallel.getEncoded(),contentStream,contentType);
    
  }
  
  class ProcessableInputStream implements CMSProcessable 
    {
      private DigestInputStream in;
      MessageDigest digester;
      byte digestResult [];

      public void write(OutputStream out) throws IOException, CMSException {
        byte b[] = new byte[8192];
        int read = in.read(b);
        while (read > 0)
        {
          out.write(b, 0, read);
          read = in.read(b);
        }
        out.close();
        in.close();
        digestResult = digester.digest();
      }

      public Object getContent() {
        return in;
      }

      public ProcessableInputStream(InputStream datain) throws NoSuchAlgorithmException, NoSuchProviderException {
        super();
        digester  = MessageDigest.getInstance("SHA-1", "BC");
        in = new DigestInputStream (datain, digester);
        digestResult = null;
      }

      public byte [] getDigest() {
        return digestResult;
      }
      
    }
  
  
  
  
  
}

