package es.caib.signatura.provider.impl.common;

import java.io.ByteArrayInputStream;
import java.io.InputStream;
import java.io.OutputStream;
import java.lang.reflect.Constructor;
import java.math.BigInteger;
import java.security.cert.CertStore;
import java.security.cert.CollectionCertStoreParameters;
import java.security.cert.X509Certificate;

import com.lowagie.text.DocumentException;
import com.lowagie.text.Element;
import com.lowagie.text.Font;
import com.lowagie.text.Image;
import com.lowagie.text.Phrase;
import com.lowagie.text.Rectangle;
import com.lowagie.text.pdf.AcroFields;
import com.lowagie.text.pdf.BarcodePDF417;
import com.lowagie.text.pdf.BaseFont;
import com.lowagie.text.pdf.PdfContentByte;
import com.lowagie.text.pdf.PdfDate;
import com.lowagie.text.pdf.PdfDictionary;
import com.lowagie.text.pdf.PdfName;
import com.lowagie.text.pdf.PdfPCell;
import com.lowagie.text.pdf.PdfPKCS7;
import com.lowagie.text.pdf.PdfPTable;
import com.lowagie.text.pdf.PdfReader;
import com.lowagie.text.pdf.PdfSignature;
import com.lowagie.text.pdf.PdfSignatureAppearance;
import com.lowagie.text.pdf.PdfStamper;
import com.lowagie.text.pdf.PdfString;
import com.lowagie.text.pdf.PdfTemplate;

import es.caib.signatura.api.SignatureException;
import es.caib.signatura.api.Signer;
import es.caib.signatura.impl.ClassLoaderFactory;
import es.caib.signatura.impl.SigDebug;
import es.caib.signatura.impl.SignaturaProperties;
import es.caib.signatura.impl.locales.Messages;

import java.security.DigestInputStream;
import java.security.MessageDigest;
import java.security.PrivateKey;
import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.Hashtable;
import java.util.Properties;
import java.util.Vector;

import org.bouncycastle.asn1.DERInteger;
import org.bouncycastle.asn1.DERObjectIdentifier;
import org.bouncycastle.asn1.DERSet;
import org.bouncycastle.asn1.cms.Attribute;
import org.bouncycastle.asn1.cms.AttributeTable;
import org.bouncycastle.asn1.ess.ESSCertID;
import org.bouncycastle.asn1.ess.SigningCertificate;
import org.bouncycastle.asn1.x509.GeneralName;
import org.bouncycastle.asn1.x509.GeneralNames;
import org.bouncycastle.asn1.x509.IssuerSerial;
import org.bouncycastle.asn1.x509.X509Name;
import org.bouncycastle.cms.CMSProcessable;
import org.bouncycastle.cms.CMSProcessableByteArray;
import org.bouncycastle.cms.CMSSignedData;
import org.bouncycastle.cms.CMSSignedDataGenerator;

public class PDFSigner {
	
	private static final int SIGNATURE_HORIZONTAL_SPLIT_LEN = 65;
	private static final int SIGNATURE_VERTICAL_SPLIT_LEN = 100;
	private static final int SIGNATURE_SPLIT_RANGE = 10;
	private static final int SIGNATURE_SPLIT_MARGIN = 5;
	
	private static int SIGNATURE_DEF_WIDTH = 260;
	private static int SIGNATURE_DEF_HEIGHT = 50;
		
	public static final String DATE_FORMAT = "dd-MM-yyyy"; //$NON-NLS-1$
	private static final int SIGNATURE_RESERVED_BYTES = 10000;
	
	private static int CERTIFY_DEF_WIDTH = 350;
	private static int CERTIFY_DEF_HEIGHT = 75;
	private static String CERTIFY_DATE_FORMAT = "dd/MM/yyyy H:m:s";
	private static int CERTIFY_FONT_SIZE = 4;
	private static final String CERTIFY_TEXT = "ÉS CÒPIA ELECTRÒNICA AUTENTICADA " +
	"DEL DOCUMENT ORIGINAL";
	
	/**
	 * @param pdfInputStream
	 * @param signedStream
	 * @param key
	 * @param chain
	 * @param textoAdicional
	 * @param position
	 * @param allowMultipleSignature
	 * @param providerName
	 * @throws Exception
	 */
	public static void sign(InputStream pdfInputStream, OutputStream signedStream,
	        PrivateKey key, X509Certificate[] chain, String textoAdicional, int position,
	        boolean allowMultipleSignature, String providerName) throws Exception {
		
		if ((position & Signer.PDF_SIGN_DEFAULT_SIGNATURE_APPERANCE) == 0 || position == Signer.PDF_SIGN_POSITION_NONE  || (position & Signer.PDF_SIGN_PDF417_SIGNATURE_APPERANCE)!=0) {
			estampadoPDF417(pdfInputStream, signedStream, key, chain, textoAdicional, position, allowMultipleSignature, providerName);
		} else {
			estampadoAdobe(pdfInputStream, signedStream, key, chain, textoAdicional, position, allowMultipleSignature, providerName);
		}
		
	}
	


	/**
	 * @param pdfInputStream
	 * @param signedStream
	 * @param key
	 * @param chain
	 * @param textoAdicional
	 * @param position
	 * @param allowMultipleSignature
	 * @param providerName
	 * @throws Exception
	 */
	public static void sign(InputStream pdfInputStream, OutputStream signedStream,
	        PrivateKey key, X509Certificate[] chain, String textoAdicional, int stampOptions,
	        float top, float left,float height, float width, float rotation,
	        boolean allowMultipleSignature, String providerName) throws Exception {
		
			if ((stampOptions & Signer.PDF_SIGN_DEFAULT_SIGNATURE_APPERANCE) == 0 || stampOptions == Signer.PDF_SIGN_POSITION_NONE || (stampOptions & Signer.PDF_SIGN_PDF417_SIGNATURE_APPERANCE)!=0) {
				estampadoPDF417(pdfInputStream, signedStream, key, chain, textoAdicional, stampOptions,top,left,height,width,rotation, allowMultipleSignature, providerName);
			} else {
				estampadoAdobe(pdfInputStream, signedStream, key, chain, textoAdicional, stampOptions,top,left,height,width,rotation, allowMultipleSignature, providerName);
			}

		
	}
	
	
	public static void certifyDigitalCopy(InputStream pdfInputStream, OutputStream signedStream,
	        PrivateKey key, X509Certificate[] chain, String url, String localidad, float x, float y, float rotation,SignaturaProperties signatureProperties, String providerName)
		throws Exception {
				
		PdfReader reader;
		
		try {
			
			reader = new PdfReader(pdfInputStream);
						
			//Miramos a qué clase debemos delegar el parseo del certificado y la instanciamos. Por defecto la de la CAIB.
			String funcionariParserClassName=signatureProperties.getPublicEmployeeParser();
			Class funcionariParserClass=ClassLoaderFactory.getFactory().getMasterClassLoader().loadClass(funcionariParserClassName);
			Constructor constructor=funcionariParserClass.getConstructor(
					new Class[] {
							X509Certificate[].class,
							boolean.class 
					}
				);
			FuncionariParsedCertificate certificate=(FuncionariParsedCertificate)constructor.newInstance(
					new Object[] {  chain,
									new Boolean(true)
					}
				);
			
			
			
			//iniciamos la firma, permitiendo que el pdf se firme más de una vez, creando una nueva revisión del documento.
			PdfStamper stp = null;
			stp = PdfStamper.createSignature(reader, signedStream, '\0', null, true);
			
			PdfSignatureAppearance sap = stp.getSignatureAppearance();
			sap.setCrypto(key, chain, null, PdfSignatureAppearance.WINCER_SIGNED);
			sap.setReason(CERTIFY_TEXT);
			
			// Sólo ponemos el sello de compulsa en la primera página
			int pageNumber = 1;

			//TODO: hacer que funcione sin corte la rotación
			// Coordenadas finales del rectángulo que define el área de
			// la SignatureAppearance
			float x2 = 0, y2 = 0;
			// Posición del template (importante para correcta rotación)
			float topTemplate = 0, leftTemplate = 0;
			
			// Tratamiento normal
			if (rotation == 0) {
				
				x2 = (x + CERTIFY_DEF_WIDTH);
				y2 = (y + CERTIFY_DEF_HEIGHT);
				
				leftTemplate = 0;
				topTemplate = 0;
			
			// Rotamos 90 grados a la derecha siendo el punto (x,y) el eje de rotación
			} else if (rotation == 90) {
				
				x2 = (x + CERTIFY_DEF_HEIGHT);
				y2 = (y + CERTIFY_DEF_WIDTH);

				leftTemplate = CERTIFY_DEF_HEIGHT;
				topTemplate = 1;
				
			// Rotamos 270 grados a la derecha siendo el punto (x,y) el eje de rotación
			} else if (rotation == 270 || rotation == -90) {
				
				x2 = (x + CERTIFY_DEF_HEIGHT);
				y2 = (y + CERTIFY_DEF_WIDTH);
				
				leftTemplate = 0;
				topTemplate = CERTIFY_DEF_WIDTH - 1;
				
			} else {
				
				throw new SignatureException("La rotación especificada no está dentro de los valores soportados [0|90|270]");
				
			}
						
			Rectangle rec = new Rectangle(x, y, x2, y2);
			sap.setVisibleSignature(rec, pageNumber, certificate.getNombre());
			sap.setAcro6Layers(true);
						
			// Layer n2 de la PdfSignatureAppearance
			PdfTemplate n2 = sap.getLayer(2);
			
			// Template donde añadiremos la tabla con los datos de compulsa
			PdfTemplate tem = PdfTemplate.createTemplate(stp.getWriter(), (x2 - x), (y2 - y));

			// La tabla
			PdfPTable table = new PdfPTable(new float[] {0.13f, 0.66f, 0.09f, 0.12f});
			
			// Tipo de fuente
			BaseFont bf = BaseFont.createFont("Helvetica", BaseFont.WINANSI, BaseFont.EMBEDDED); //$NON-NLS-1$
			Font font = new Font(bf);
			font.setSize(CERTIFY_FONT_SIZE);

			// Fila 1
			PdfPCell cell = new PdfPCell(new Phrase(CERTIFY_TEXT, font));
			cell.setColspan(4);
			cell.setHorizontalAlignment(Element.ALIGN_CENTER);
			table.addCell(cell);
			
			// Fila 2
			cell = new PdfPCell(new Phrase("Signat per", font));
			cell.setHorizontalAlignment(Element.ALIGN_LEFT);
			table.addCell(cell);
			
			String nombreCargo = certificate.getNombre() + " " + certificate.getApellidos() + 
				" - " + certificate.getCargo();
			cell = new PdfPCell(new Phrase(nombreCargo, font));
			cell.setColspan(3);
			cell.setHorizontalAlignment(Element.ALIGN_LEFT);
			table.addCell(cell);
			
			// Fila 3
			cell = new PdfPCell(new Phrase("Centre directiu adscrit", font));
			cell.setHorizontalAlignment(Element.ALIGN_LEFT);
			table.addCell(cell);
			
			cell = new PdfPCell(new Phrase(certificate.getCentroDirectivo(), font));
			cell.setHorizontalAlignment(Element.ALIGN_LEFT);
			table.addCell(cell);
			
			cell = new PdfPCell(new Phrase("Data d'emissió", font));
			cell.setHorizontalAlignment(Element.ALIGN_LEFT);
			table.addCell(cell);
			
			Date fecha = sap.getSignDate().getTime();
			DateFormat formatter = new SimpleDateFormat(CERTIFY_DATE_FORMAT);
            String fechaFormateada = formatter.format(fecha);
			
			cell = new PdfPCell(new Phrase(fechaFormateada, font));
			cell.setHorizontalAlignment(Element.ALIGN_LEFT);
			table.addCell(cell);
			
			// Fila 4
			cell = new PdfPCell(new Phrase("Conselleria", font));
			cell.setHorizontalAlignment(Element.ALIGN_LEFT);
			table.addCell(cell);
			
			cell = new PdfPCell(new Phrase(certificate.getConsejeria(), font));
			cell.setHorizontalAlignment(Element.ALIGN_LEFT);
			table.addCell(cell);
			
			cell = new PdfPCell(new Phrase("Lloc d'emissió", font));
			cell.setHorizontalAlignment(Element.ALIGN_LEFT);
			table.addCell(cell);
			
			cell = new PdfPCell(new Phrase(localidad, font));
			cell.setHorizontalAlignment(Element.ALIGN_LEFT);
			table.addCell(cell);
			
			// start Código de barras
			String line = url;
							
			float pdf417_X = 20;
			float pdf417_Y = 20;
										
			BarcodePDF417 pdf417 = new BarcodePDF417();
			pdf417.setText(url);
			Image image = pdf417.getImage();
											
			int SIGNATURE_SPLIT_LEN = SIGNATURE_HORIZONTAL_SPLIT_LEN;
			
//			int SIGNATURE_SPLIT_LEN = 0;
//			if (height > width)
//				SIGNATURE_SPLIT_LEN = (position == Signer.PDF_SIGN_POSITION_LEFT || position == Signer.PDF_SIGN_POSITION_RIGHT) ? SIGNATURE_VERTICAL_SPLIT_LEN : SIGNATURE_HORIZONTAL_SPLIT_LEN;
//			else
//				SIGNATURE_SPLIT_LEN = (position == Signer.PDF_SIGN_POSITION_LEFT || position == Signer.PDF_SIGN_POSITION_RIGHT) ? SIGNATURE_HORIZONTAL_SPLIT_LEN : SIGNATURE_VERTICAL_SPLIT_LEN;
			
			Vector lines = new Vector();
			int counter = 0;
			
			while (counter != -1) {
				
				if ( counter + SIGNATURE_SPLIT_LEN < line.length() ) {
					
					int prev = line.substring(counter,counter + SIGNATURE_SPLIT_LEN).lastIndexOf(" "); //$NON-NLS-1$
					String next_string = line.substring(counter + SIGNATURE_SPLIT_LEN, line.length());
					int next = next_string.indexOf(" "); //$NON-NLS-1$
					//prev [-1,SPLIT_LEN]
					//next[-1,inf]

					int splitPos = prev + 1; 
					//splitpos [-1,SPLIT_LEN+1]

					if (next < SIGNATURE_SPLIT_RANGE && next != -1) {
						
						splitPos = SIGNATURE_SPLIT_LEN + next;
						//next[0,inf]
						//splitpos [0,SPLIT_LEN+SPLIT_RANGE]
						
					} else if (prev == -1 && next > SIGNATURE_SPLIT_RANGE) {
						
						splitPos = SIGNATURE_SPLIT_LEN;
						//prev [0,SPLIT_LEN]
						//splitpos [0,SPLIT_LEN+SPLIT_RANGE]
						
					} else if(prev == -1 && next == -1) {
						
						splitPos = SIGNATURE_SPLIT_LEN;
						
					}
					
					lines.add( line.substring(counter, counter + splitPos) );
					counter += splitPos;
					
				} else {
					
					lines.add(line.substring(counter, line.length()));
					counter = -1;
				
				}
			
			} // end while

			image.setAbsolutePosition(pdf417_X, pdf417_Y);
			image.scalePercent(100);

			// Fila 5
			cell = new PdfPCell(image);
			cell.setColspan(4);
			cell.setPadding(3);
			cell.setHorizontalAlignment(Element.ALIGN_CENTER);
			table.addCell(cell);
			
			// Fila 6
			cell = new PdfPCell(new Phrase(url, font));
			cell.setColspan(4);
			cell.setHorizontalAlignment(Element.ALIGN_CENTER);
			table.addCell(cell);
			// end código de barras

						
			// Establecemos el ancho total de la tabla
			table.setTotalWidth(Math.abs(CERTIFY_DEF_WIDTH)); // 1px = grosor del borde por defecto
									
			// Y la escribimos en el template a añadir al layer n2
			table.writeSelectedRows(0, -1, 0, CERTIFY_DEF_HEIGHT - 1, tem);  // 1px = grosor del borde por defecto

			float angle  = (float)(-rotation * (Math.PI / 180));
			float xScale = (float)(Math.cos(angle));
			float yScale = (float)(Math.cos(angle));
			float xRote  = (float)(-Math.sin(angle));
			float yRote  = (float)(Math.sin(angle)); 
			
			n2.addTemplate(tem, xScale, xRote, yRote, yScale, leftTemplate, topTemplate);
			
			// DEBUG
			/*
			System.out.println("Ancho de página: " + reader.getPageSize(pageNumber).getWidth());
			System.out.println("Alto de página: " + reader.getPageSize(pageNumber).getHeight());
			System.out.println("Creando rectángulo en: ");
			System.out.println("(x=" + x + ",y=" + y + ")");
			System.out.println("(x2=" + x2 + ",y2=" + y2 + ")");
			System.out.println("Medidas (WxH): " + Math.abs(x2 - x) + "x" + Math.abs(y2 - y));
			System.out.println("leftTemplate: " + leftTemplate);
			System.out.println("topTemplate: " + topTemplate);
			System.out.println("Ancho del Template: " + tem.getWidth());
			System.out.println("Alto del Template: " + tem.getHeight());
			*/
		
			/**
			 * Código para la external signature
			 */

			PdfSignature dic = new PdfSignature(PdfName.ADOBE_PPKMS, PdfName.ADBE_PKCS7_SHA1);
	        dic.setDate(new PdfDate(sap.getSignDate()));
	        dic.setName(PdfPKCS7.getSubjectFields((X509Certificate)chain[0]).getField("CN"));
	        
	        if (sap.getReason() != null)
	            dic.setReason(sap.getReason());
	        if (sap.getLocation() != null)
	            dic.setLocation(sap.getLocation());
	        
	        sap.setCryptoDictionary(dic);
	        int csize = SIGNATURE_RESERVED_BYTES;
	        HashMap exc = new HashMap();
	        exc.put(PdfName.CONTENTS, new Integer(csize * 2 + 2));
	        sap.preClose(exc);

	        MessageDigest md = MessageDigest.getInstance("SHA1");
	        InputStream s = sap.getRangeStream();
	        int read = 0;
	        byte[] buff = new byte[8192];
	        while ((read = s.read(buff, 0, 8192)) > 0) {
	            md.update(buff, 0, read);
	        }

	        CMSSignedDataGenerator generator = new CMSSignedDataGenerator();
	        
	        // Obtenemos los atributos firmados y sin firmar, para añadirlos al
			// método addSigner()
			AttributeTable signedAttributes = getSignedAttributes((X509Certificate)chain[0]);
			AttributeTable unsignedAttributes = getUnsignedAttributes((X509Certificate)chain[0]);
			
			//Añadimos el Signer que utilizaremos en la firma
			generator.addSigner(key, (X509Certificate)chain[0], CMSSignedDataGenerator.DIGEST_SHA1, signedAttributes, unsignedAttributes);	        

	        ArrayList list = new ArrayList();
	        for (int i = 0; i < chain.length; i++) {
	            list.add(chain[i]);
	        }
	        
	        CertStore chainStore = CertStore.getInstance("Collection", new CollectionCertStoreParameters(list));
	        generator.addCertificatesAndCRLs(chainStore);
	        CMSProcessable content = new CMSProcessableByteArray(md.digest());
	        CMSSignedData signedData = generator.generate(content, true, providerName);
	        
	        byte[] pk = signedData.getEncoded();

	        byte[] outc = new byte[csize];

	        PdfDictionary dic2 = new PdfDictionary();

	        System.arraycopy(pk, 0, outc, 0, pk.length);

	        dic2.put(PdfName.CONTENTS, new PdfString(outc).setHexWriting(true));
	        sap.close(dic2);

			/**
			 * Fin código para external signature
			 */
			
		} catch (DocumentException e) {
			
			throw new SignatureException (Messages.getString("PDFSigner.error_firmando_pdf"), e); //$NON-NLS-1$
		
		}
		
	}
	
	private static void estampadoAdobe(InputStream pdfInputStream, OutputStream signedStream, 
			PrivateKey key, X509Certificate[] chain, String reason, int position, boolean allowMultipleSignature, String providerName)  
		throws Exception {
		
		PdfReader reader;

		reader = new PdfReader(pdfInputStream);
		int numberOfPages = reader.getNumberOfPages();
		ParsedCertificateImpl certificate = new ParsedCertificateImpl(chain, true);
		int pageNumber =	((position & Signer.PDF_SIGN_PAGE_LAST)==0 )?	1	:	numberOfPages;
		int top = 0;
		int left = 0;
	
		


		estampadoAdobe(reader, signedStream, key, certificate, chain, reason, position, top, left,SIGNATURE_DEF_HEIGHT,SIGNATURE_DEF_WIDTH,0, allowMultipleSignature, providerName,true);
	}

	
	private static void estampadoAdobe(InputStream pdfInputStream, OutputStream signedStream, 
			PrivateKey key, X509Certificate[] chain, String reason, int stampOptions,float top, float left,float height, float width,float rotation, boolean allowMultipleSignature, String providerName)  
		throws Exception {
		
		PdfReader reader;

		reader = new PdfReader(pdfInputStream);
		int numberOfPages = reader.getNumberOfPages();
		ParsedCertificateImpl certificate = new ParsedCertificateImpl(chain, true);
		int pageNumber =	((stampOptions & Signer.PDF_SIGN_PAGE_LAST)==0 )?	1	:	numberOfPages;

		
		Rectangle pageSize = reader.getPageSize(pageNumber);
		float p_width = pageSize.getWidth();
		float p_height = pageSize.getHeight();			
		
		if(left<0)	
			left = p_width +left;
		
		if(top<0)
			top= p_height + top;

		estampadoAdobe(reader, signedStream, key, certificate, chain, reason, stampOptions, top, left,height,width,rotation,allowMultipleSignature, providerName,false);
	}
	
	
	private static void estampadoAdobe(PdfReader reader, OutputStream signedStream, 
			PrivateKey key, ParsedCertificateImpl certificate,X509Certificate[] chain, String reason, int stampOptions,float top, float left,float height, float width,float rotation, boolean allowMultipleSignature, String providerName, boolean adjustSize)  
		throws Exception {	
			
		try {
			
			int numberOfPages = reader.getNumberOfPages();

			AcroFields acroFields = reader.getAcroFields();
			ArrayList names = acroFields.getSignatureNames();
			
			if (names.size() > 0 && !allowMultipleSignature) {
				throw new Exception(Messages.getString("PDFSigner.error_documento_ya_firmado")); //$NON-NLS-1$
			}

			// Se sella el documento PDF;
	
			PdfStamper stp = null;
			
			if (allowMultipleSignature)
				stp = PdfStamper.createSignature(reader, signedStream, '\0', null, true);
			else
				stp = PdfStamper.createSignature(reader, signedStream, '\0');
			
			PdfSignatureAppearance sap = stp.getSignatureAppearance();
			sap.setCrypto(key, chain, null, PdfSignatureAppearance.WINCER_SIGNED);
			sap.setReason(reason);	
			
			int pageNumber =	((stampOptions & Signer.PDF_SIGN_PAGE_LAST)==0 )?	1	:	numberOfPages;
			Rectangle pageSize = reader.getPageSize(pageNumber);
			float pageWidth = pageSize.getWidth();
			float pageHeight = pageSize.getHeight();
			
			//recalculamos la anchura del texto para calcular la posición:
			String texto[] = new String[] {
				Messages.getString("PDFSigner.sello_adobe_firmado_por") + certificate.getCommonName(), //$NON-NLS-1$
				Messages.getString("PDFSigner.sello_adobe_fecha") + sap.getSignDate().getTime().toLocaleString(), //$NON-NLS-1$
				Messages.getString("PDFSigner.sello_adobe_motivo") + sap.getReason() //$NON-NLS-1$
			};
			
			if(adjustSize){ //ajustamos el tamaño en función de mayúsculas/minúsculas
				for (int i = 0; i < texto.length; i++) {
					int text_width = 0;
					for (int j = 0; j < texto[i].length(); j++){
						char c=texto[i].charAt(j);
						if (c>96 && c<123) text_width += 4;   	//minusculas
						else text_width += 6; 					//otros
					}
					if( text_width + 30 > width){
						width = text_width + 30;				
					}
				}
			}		
			
			if ((stampOptions & Signer.PDF_SIGN_POSITION_LEFT) == Signer.PDF_SIGN_POSITION_LEFT) {
				//izquierda
				left = 20;
			} else if ((stampOptions & Signer.PDF_SIGN_POSITION_RIGHT) == Signer.PDF_SIGN_POSITION_RIGHT){
				//derecha
				left = (int)pageWidth - width - 20;
				
			} else if ((stampOptions & Signer.PDF_SIGN_POSITION_TOP) == Signer.PDF_SIGN_POSITION_TOP || (stampOptions & Signer.PDF_SIGN_POSITION_BOTTOM) == Signer.PDF_SIGN_POSITION_BOTTOM){
				//centro
				left = (int)(pageWidth - width)/2;
			}

			if ((stampOptions & Signer.PDF_SIGN_POSITION_TOP) == Signer.PDF_SIGN_POSITION_TOP) {
				//arriba
				top = (int)pageHeight - 20;
			} else if((stampOptions & Signer.PDF_SIGN_POSITION_BOTTOM) == Signer.PDF_SIGN_POSITION_BOTTOM){
				//abajo
				top = 20 + SIGNATURE_DEF_HEIGHT;
			} else if ((stampOptions & Signer.PDF_SIGN_POSITION_LEFT) == Signer.PDF_SIGN_POSITION_LEFT || (stampOptions & Signer.PDF_SIGN_POSITION_RIGHT) == Signer.PDF_SIGN_POSITION_RIGHT){
				//centro
				top = (int)(pageHeight - SIGNATURE_DEF_HEIGHT)/2;
			}		
			
			Rectangle rec = new Rectangle(left,top-height,left+width,top);
			BaseFont bf = BaseFont.createFont(BaseFont.HELVETICA, BaseFont.WINANSI, BaseFont.EMBEDDED);

			sap.setVisibleSignature(rec, pageNumber,certificate.getName());
			sap.setAcro6Layers(true);
			
			com.lowagie.text.pdf.PdfTemplate n2 = sap.getLayer(2);

			n2.beginText();
			
			bf = BaseFont.createFont("Helvetica", BaseFont.WINANSI, false); //$NON-NLS-1$
			
			//Presentem els texts
			n2.setColorStroke(java.awt.Color.BLACK);
			n2.setFontAndSize(bf, 8);		
			for (int i = 0; i < texto.length; i++){
				n2.setTextMatrix(30,n2.getHeight()-(i+1)*10);
				n2.showText(texto[i]);
			}
			
			n2.endText();
			
			/**
			 * Código para la external signature
			 */

			PdfSignature dic = new PdfSignature(PdfName.ADOBE_PPKMS, PdfName.ADBE_PKCS7_SHA1);
	        dic.setDate(new PdfDate(sap.getSignDate()));
	        dic.setName(PdfPKCS7.getSubjectFields((X509Certificate)chain[0]).getField("CN"));
	        
	        if (sap.getReason() != null)
	            dic.setReason(sap.getReason());
	        if (sap.getLocation() != null)
	            dic.setLocation(sap.getLocation());
	        
	        sap.setCryptoDictionary(dic);
	        int csize = SIGNATURE_RESERVED_BYTES;
	        HashMap exc = new HashMap();
	        exc.put(PdfName.CONTENTS, new Integer(csize * 2 + 2));
	        sap.preClose(exc);

	        MessageDigest md = MessageDigest.getInstance("SHA1");
	        InputStream s = sap.getRangeStream();
	        int read = 0;
	        byte[] buff = new byte[8192];
	        while ((read = s.read(buff, 0, 8192)) > 0) {
	            md.update(buff, 0, read);
	        }

	        CMSSignedDataGenerator generator = new CMSSignedDataGenerator();
	        
	        // Obtenemos los atributos firmados y sin firmar, para añadirlos al
			// método addSigner()
			AttributeTable signedAttributes = getSignedAttributes((X509Certificate)chain[0]);
			AttributeTable unsignedAttributes = getUnsignedAttributes((X509Certificate)chain[0]);
			
			//Añadimos el Signer que utilizaremos en la firma
			generator.addSigner(key, (X509Certificate)chain[0], CMSSignedDataGenerator.DIGEST_SHA1, signedAttributes, unsignedAttributes);	        

	        ArrayList list = new ArrayList();
	        for (int i = 0; i < chain.length; i++) {
	            list.add(chain[i]);
	        }
	        
	        CertStore chainStore = CertStore.getInstance("Collection", new CollectionCertStoreParameters(list));
	        generator.addCertificatesAndCRLs(chainStore);
	        CMSProcessable content = new CMSProcessableByteArray(md.digest());
	        CMSSignedData signedData = generator.generate(content, true, providerName);
	        
	        byte[] pk = signedData.getEncoded();

	        byte[] outc = new byte[csize];

	        PdfDictionary dic2 = new PdfDictionary();

	        System.arraycopy(pk, 0, outc, 0, pk.length);

	        dic2.put(PdfName.CONTENTS, new PdfString(outc).setHexWriting(true));
	        sap.close(dic2);

			/**
			 * Fin código para external signature
			 */
	        
			
		} catch (DocumentException e) {
			
			throw new SignatureException (Messages.getString("PDFSigner.error_firmando_pdf"), e); //$NON-NLS-1$
		
		}

	} // end estampadoAdobe()

	
	private static void estampadoPDF417(InputStream pdfInputStream, OutputStream signedStream, PrivateKey key, X509Certificate[] chain, String url, int stampOptions, boolean allowMultipleSignature, String providerName) throws Exception {
		PdfReader reader;
		reader = new PdfReader(pdfInputStream);
		
		float x,y,height=500,width=(SIGNATURE_HORIZONTAL_SPLIT_LEN+SIGNATURE_SPLIT_RANGE)*15,rotation;
		
		switch (stampOptions) {
			
			case (Signer.PDF_SIGN_POSITION_RIGHT):
				x = -20;
				y = 20;
				rotation = 90;
				break;
			case (Signer.PDF_SIGN_POSITION_TOP):
				x = 20;
				y = -50;
				rotation = 0;
				break;
			case (Signer.PDF_SIGN_POSITION_BOTTOM):
				x = 20;
				y = 30;
				rotation = 0;
				break;
			default: 
				//case (Signer.PDF_SIGN_POSITION_LEFT):
				x = 40;
				y = 20;
				rotation = 90;
				break;
		}
		
		estampadoPDF417(reader, signedStream, key, chain, url, stampOptions,y,x,height,width,rotation, allowMultipleSignature, providerName);
	}
	
	private static void estampadoPDF417(InputStream pdfInputStream, OutputStream signedStream,
			PrivateKey key, X509Certificate[] chain, String url, int stampOptions,float top, float left, float height, float width,float rotation, boolean allowMultipleSignature, String providerName)
		throws Exception {
		PdfReader reader;
		reader = new PdfReader(pdfInputStream);
		

		
		estampadoPDF417(reader, signedStream, key, chain, url, stampOptions,top,left,height,width,rotation, allowMultipleSignature, providerName);
	}
	
	private static void estampadoPDF417(PdfReader reader, OutputStream signedStream,
			PrivateKey key, X509Certificate[] chain, String url, int position,float top, float left, float height, float width, float rotation, boolean allowMultipleSignature, String providerName)
		throws Exception {
		

		try {

			int numberOfPages = reader.getNumberOfPages();

			AcroFields acroFields = reader.getAcroFields();
			ArrayList names = acroFields.getSignatureNames();
			
			if (names.size() > 0 && !allowMultipleSignature) {
				throw new Exception(Messages.getString("PDFSigner.error_documento_ya_firmado")); //$NON-NLS-1$
			}

			ParsedCertificateImpl certificate = new ParsedCertificateImpl(chain, true);
			// Se sella el documento PD;
			String signName =certificate.getName();
			SimpleDateFormat simpleDateFormat = new SimpleDateFormat(DATE_FORMAT);
	
			BaseFont bf = BaseFont.createFont(BaseFont.HELVETICA, BaseFont.WINANSI, BaseFont.EMBEDDED);
			String  line = signName + " " + simpleDateFormat.format(new Date()) + " " + url; //$NON-NLS-1$ //$NON-NLS-2$

			PdfStamper stamp;
			
			// Comprobamos si queremos que el PDF contenga múltiples firmas
			if (allowMultipleSignature)
				stamp = PdfStamper.createSignature(reader, signedStream, '\0', null, true);
			else
				stamp = PdfStamper.createSignature(reader, signedStream, '\0');
			
			PdfSignatureAppearance sap = stamp.getSignatureAppearance();
			sap.setCrypto(key, chain, null, PdfSignatureAppearance.WINCER_SIGNED);

			// Template donde añadiremos la tabla con los datos de compulsa
			
			if(position != Signer.PDF_SIGN_POSITION_NONE){
				int textAlign = PdfContentByte.ALIGN_LEFT;
				

				BarcodePDF417 pdf417 = new BarcodePDF417();
				pdf417.setText(url);
				Image image = pdf417.getImage();
				if(width<image.getWidth()) throw new SignatureException("Width too small, minumim: "+image.getWidth());
				if(height<image.getHeight()) throw new SignatureException("Height too small, minumim: "+image.getHeight());					
				
				for (int i = 1; i <=numberOfPages; i++) {
					PdfContentByte over = stamp.getOverContent(i);
					PdfTemplate template = PdfTemplate.createTemplate(stamp.getWriter(), width, height);
					
					Rectangle pageSize = reader.getPageSize(i);
					float pageWidth = pageSize.getWidth();
					float pageHeight = pageSize.getHeight();
					
					int SIGNATURE_SPLIT_LEN=0;
					SIGNATURE_SPLIT_LEN=(int)(width-image.getWidth())/8;
					

					Vector lines=new Vector();
					int counter=0;
					boolean ampliat=false;
					
					while(counter!=-1){
						if(counter+SIGNATURE_SPLIT_LEN<line.length()){
							int prev=line.substring(counter,counter+SIGNATURE_SPLIT_LEN).lastIndexOf(" "); //$NON-NLS-1$
							String next_string=line.substring(counter+SIGNATURE_SPLIT_LEN,line.length());
							int next=next_string.indexOf(" "); //$NON-NLS-1$
							//prev [-1,SPLIT_LEN]
							//next[-1,inf]
		
							int splitPos=prev+1; 
							//splitpos [-1,SPLIT_LEN+1]
		
							if(!ampliat && next<SIGNATURE_SPLIT_RANGE && next!=-1){
								splitPos=SIGNATURE_SPLIT_LEN+next;
								SIGNATURE_SPLIT_LEN+=next; //per a què totes les línies quedin iguals
								ampliat=true;
								//next[0,inf]
								//splitpos [0,SPLIT_LEN+SPLIT_RANGE]
								
							}else if(prev==-1 && next>SIGNATURE_SPLIT_RANGE ){
								splitPos=SIGNATURE_SPLIT_LEN;
								//prev [0,SPLIT_LEN]
								//splitpos [0,SPLIT_LEN+SPLIT_RANGE]
							}else if(prev==-1 && next==-1){
								splitPos=SIGNATURE_SPLIT_LEN;
							}
							
							lines.add(line.substring(counter,counter+splitPos));
							if((counter+splitPos)<line.length() && line.charAt(counter+splitPos)==' ') counter++;
							counter+=splitPos; //+1= espai
							
							
						}else{
							lines.add(line.substring(counter,line.length()));
							counter=-1;
						}
					}
		
					float pdf417_X =0;
					float pdf417_Y =0;
					float text_X = image.getWidth()+20;
					float text_Y = 0;
		

					image.setAbsolutePosition(pdf417_X, ((lines.size())*(10+SIGNATURE_SPLIT_MARGIN))/2.0f-(image.getHeight()/2.0f));
					template.addImage(image);
		
					counter=0;
					
					
					//Imprimimos las líneas de la firma
					for(counter=0;counter<lines.size();counter++){
				
						template.beginText();
						template.setFontAndSize(bf, 10);
						float x= text_X;
						float y=text_Y+((lines.size()-1)-counter)*(10+SIGNATURE_SPLIT_MARGIN);
						template.showTextAligned(textAlign, (String)lines.get(counter),x ,y ,0);
						template.endText();		



					}					
					
					//default rotation and scale values
					float angle  = (float)(-rotation * (Math.PI / 180));
					float xScale = (float)(Math.cos(angle));
					float yScale = (float)(Math.cos(angle));
					float xRote  = (float)(-Math.sin(angle));
					float yRote  = (float)(Math.sin(angle)); 

					//cos and sin doesn't give exact values, so text is not sharp
					//force exact values to sharp text
					if(-rotation==90 || -rotation==-270){
						xScale = 0.0f;  //cos
						yRote  = 1.0f;  //-sin
						xRote  = -1.0f;  //sin
						yScale = 0.0f; 	//con			
					}else if(-rotation==180 || -rotation==-180){
						xScale = -1.0f;  //cos
						yRote  = 0.0f;  //-sin
						xRote  = 0.0f;  //sin
						yScale  = -1.0f; 	//con			
						
					}else if(-rotation==270 || -rotation==-90){
						xScale = 0.0f;  //cos
						yRote = -1.0f;  //-sin
						xRote  = 1.0f;  //sin
						yScale   = 0.0f; 	//con			
						
					}

							
					
					if(left<0)	
						left = pageWidth +left;
					
					if(top<0)
						top= pageHeight + top;
					
					over.addTemplate(template, xScale, xRote, yRote, yScale, left, top);
				}


			}
			
			/**
			 * Código para la external signature
			 */
			
			PdfSignature dic = new PdfSignature(PdfName.ADOBE_PPKMS, PdfName.ADBE_PKCS7_SHA1);
	        dic.setDate(new PdfDate(sap.getSignDate()));
	        dic.setName(PdfPKCS7.getSubjectFields((X509Certificate)chain[0]).getField("CN"));
	        
	        if (sap.getReason() != null)
	            dic.setReason(sap.getReason());
	        if (sap.getLocation() != null)
	            dic.setLocation(sap.getLocation());
	        
	        sap.setCryptoDictionary(dic);
	        int csize = SIGNATURE_RESERVED_BYTES;
	        HashMap exc = new HashMap();
	        exc.put(PdfName.CONTENTS, new Integer(csize * 2 + 2));
	        sap.preClose(exc);

	        MessageDigest md = MessageDigest.getInstance("SHA1");
	        InputStream s = sap.getRangeStream();
	        int read = 0;
	        byte[] buff = new byte[8192];
	        while ((read = s.read(buff, 0, 8192)) > 0) {
	            md.update(buff, 0, read);
	        }

	        CMSSignedDataGenerator generator = new CMSSignedDataGenerator();
	        
	        // Obtenemos los atributos firmados y sin firmar, para añadirlos al
			// método addSigner()
			AttributeTable signedAttributes = getSignedAttributes((X509Certificate)chain[0]);
			AttributeTable unsignedAttributes = getUnsignedAttributes((X509Certificate)chain[0]);
			
			//Añadimos el Signer que utilizaremos en la firma
			generator.addSigner(key, (X509Certificate)chain[0], CMSSignedDataGenerator.DIGEST_SHA1, signedAttributes, unsignedAttributes);	        

	        ArrayList list = new ArrayList();
	        for (int i = 0; i < chain.length; i++) {
	            list.add(chain[i]);
	        }
	        
	        CertStore chainStore = CertStore.getInstance("Collection", new CollectionCertStoreParameters(list));
	        generator.addCertificatesAndCRLs(chainStore);
	        CMSProcessable content = new CMSProcessableByteArray(md.digest());
	        CMSSignedData signedData = generator.generate(content, true, providerName);
	        
	        byte[] pk = signedData.getEncoded();

	        byte[] outc = new byte[csize];

	        PdfDictionary dic2 = new PdfDictionary();

	        System.arraycopy(pk, 0, outc, 0, pk.length);

	        dic2.put(PdfName.CONTENTS, new PdfString(outc).setHexWriting(true));
	        sap.close(dic2);
			
	        /**
			 * Fin código para la external signature
			 */
				
		} catch (DocumentException e) {
			throw new SignatureException (Messages.getString("PDFSigner.error_firmando_pdf"), e); //$NON-NLS-1$
		}
		
	}

	/**
	 * Devuelve una tabla con los atributos firmados
	 * (por ahora sólo el Signing Certificate)
	 * @param cert Certificado X509 del cual obtendremos la información
	 */
	private static AttributeTable getSignedAttributes(X509Certificate cert) throws Exception {
		
		// Obtenemos la forma codificada del certificado.
		// Al ser X.509 se obtendrá en ASN.1 DER
		byte encodedCert[] = cert.getEncoded();
		
		// Calculamos el hash del certificado
		ByteArrayInputStream baInputStream = new ByteArrayInputStream(encodedCert);
		MessageDigest digester = MessageDigest.getInstance("SHA-1", "BC");
		DigestInputStream diInputStream = new DigestInputStream(baInputStream, digester);
		
		byte digestResult[] = null;
        byte b[] = new byte[8192];
        
        for (int read = diInputStream.read(b); read > 0; read = diInputStream.read(b))
        	;
        
        diInputStream.close();
        baInputStream.close();
        digestResult = digester.digest();
        
        // Obtenemos el campo GeneralNames, a partir de un GeneralName (necesario para el IssuerSerial)
        // Obtenemos el número de serie del certificado (necesario para ESSCertID)
        // Obtenemos el IssuerSerial (necesario para ESSCertID)
        // Obtenemos el ESSCertID (necesario para SigningCertificate)
        // obtenemos el SigningCertificate
        GeneralName generalName = new GeneralName(new X509Name(cert.getIssuerX500Principal().getName()));
        GeneralNames generalNames = new GeneralNames(generalName);
        BigInteger serialNumber = cert.getSerialNumber();
        IssuerSerial issuerSerial = new IssuerSerial(generalNames, new DERInteger(serialNumber.intValue()));
		ESSCertID essCertID = new ESSCertID(digestResult, issuerSerial);
		SigningCertificate signingCertificate = new SigningCertificate(essCertID);
		
		// Creamos una Hashtable para introducir el nuevo valor
		// en los atributos firmados
		Hashtable table = new Hashtable();
		
		Attribute att = new Attribute(
				new DERObjectIdentifier("1.2.840.113549.1.9.16.2.12"), 
				new DERSet(signingCertificate)
		);

		table.put("signingCertificate", att);
		
		return new AttributeTable(table);
				
	}
	
	/**
	 * Devuelve una tabla con los atributos firmados
	 * (por ahora sólo el Signing Certificate)
	 * @param cert Certificado X509 del cual obtendremos la información
	 */
	private static AttributeTable getUnsignedAttributes(X509Certificate cert) {		
		
		return new AttributeTable(new Hashtable());
				
	}
	
	

}
