package es.caib.signatura.provider.impl.common;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.security.cert.CertificateEncodingException;
import java.security.cert.X509Certificate;
import java.util.Date;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Properties;
import java.util.Set;
import java.util.Vector;

import es.caib.signatura.api.ParsedCertificate;
import es.caib.signatura.api.Signature;
import es.caib.signatura.api.SignatureCertNotFoundException;
import es.caib.signatura.api.SignatureException;
import es.caib.signatura.api.SignaturePrivKeyException;
import es.caib.signatura.api.SignatureProviderException;
import es.caib.signatura.api.SignatureTimestampException;
import es.caib.signatura.impl.SigDebug;
import es.caib.signatura.impl.LocalSignerProviderInterface;
import es.caib.signatura.impl.SignaturaProperties;
import es.caib.signatura.impl.ValidadorProxy;

public abstract class AbstractSigner implements LocalSignerProviderInterface {

	// Nombre del provider CAIB
	protected String providerName = "";
	// Lista de certificados procesados como no válidos
	protected static Vector notValidCerts = new Vector();
	
	/**
	 * Constructor: se llama a initialize(), el cual deberá ser implementado en cada provider.
	 * @throws SignatureException 
	 */
	public AbstractSigner() throws SignatureException {
		
		try {
			
			initialize();
			
		} catch (SignatureProviderException e) {
			
			// Si es un error de que no existe el keystore por defecto de BC, no mostramos stacktrace
			if (providerName.equals("BC") && e.getMessage().contains(".keystore")) {
				
				// Propagamos mensaje conteniendo el error del keystore
				throw new SignatureException(e.getMessage(), e.getCause());
				
			} else {
				
				e.printStackTrace();
				throw new SignatureException("No se pudo cargar el provider " + providerName);
				
			}
			
		}
		
	}

	/**
	 * Método que obtiene el alias del elemento almacenado a partir de su Common Name.
	 * @param cn
	 * @param recognized
	 * @return
	 * @throws SignatureException
	 */
	protected String getAliasFromCN(String cn, boolean recognized) throws SignatureException {
		
		String aliases[] = getAliases();
		ParsedCertificateImpl[] parsedCerts = getParsedCertificates(aliases);
		ParsedCertificateImpl[] filteredCerts = filter(parsedCerts, recognized);
		
		for (int i = 0; i < filteredCerts.length; i++) {
			if (cn != null && cn.equals(filteredCerts[i].getCommonName()))
				return filteredCerts[i].getAlias();
		}
		
		throw new SignatureException(providerName + "No se ha encontrado el alias para el certificado " + cn);
		
	}
	
	/**
	 * Método que se encarga de filtrar los certificados que se le pasan (validez, uso,
	 * comprobación de si se encuentran en un dispositivo seguro y eliminación de duplicados (mismo Common Name).
	 * @param parsedCerts
	 * @param needsRecognized
	 * @return
	 */
	protected ParsedCertificateImpl[] filter(ParsedCertificateImpl[] parsedCerts, boolean needsRecognized) {
		
		Vector temp = new Vector();
		Vector out = new Vector();
		
		for (int i = 0; i < parsedCerts.length; i++) {
			
			if (filterValidity(parsedCerts[i]) && filterUsage(parsedCerts[i], needsRecognized)) {
			
				if ( isInSecureDevice(parsedCerts[i].getAlias()) ) {
				
					temp.add(parsedCerts[i]);
					
				} else {
					
					// Si no está en un dispositivo seguro pero no debe ser reconocido, lo aceptamos
					if ( !needsRecognized ) {
						
						temp.add(parsedCerts[i]);
						
					}
					
				}
			
			}
				
		}

		out = filterDuplicated(temp);
		
		return (ParsedCertificateImpl[])out.toArray(new ParsedCertificateImpl[out.size()]);

	}
	
	/**
	 * 
	 * @param parsed
	 * @return
	 */
	protected boolean filterValidity(ParsedCertificateImpl parsed) {
				
		switch (parsed.getIsInValidPeriod()) {
		
			case 0:
				return true;
			
			case -1:
				System.out.println(providerName + " Certificat " + parsed.getCommonName() 
						+ " rebutjat, el certificat encara no és vàlid.");
				notValidCerts.add(parsed);
				return false;
				
			case 1:
				System.out.println(providerName + " Certificat " + parsed.getCommonName() 
						+ " rebutjat, el certificat ha caducat.");
				notValidCerts.add(parsed);
				return false;
				
			default:
				System.out.println(providerName + " Certificat " + parsed.getCommonName() 
						+ " rebutjat. Motiu:" + parsed.getIsInValidPeriod());
				notValidCerts.add(parsed);
				return false;
					
		}
				
	}
	
	/**
	 * 
	 * @param parsed
	 * @return
	 */
	protected boolean filterUsage(ParsedCertificateImpl parsed, boolean needsRecognized) {
		
		ValidadorProxy validador = new ValidadorProxy();
								
		if ( parsed.isRecognized () && needsRecognized ||
				parsed.isAdvanced() && !needsRecognized ) {
			
			if ( validador.isEnDesenvolupament() ) {
		    	
				return true;
		    
			} else {
		    	
		    	if ( !parsed.isTest() ) {
		    		
		    		return true;
		    		
		    	} else {
		    		
		    		return false;
		    		
		    	}
		    	
		    }
			
		} else {
			
			System.out.println(providerName + " Certificat " + parsed.getCommonName() 
						+ " rebutjat, buscant " + (needsRecognized ? "reconeguts" : "avançats") 
						+ " i el certificat no ho es." );
			
			return false;
			
		}
					
	}
	
	/**
	 * 
	 * @param parsedCerts
	 * @return
	 */
	protected Vector filterDuplicated(Vector parsedCerts) {
		
		// HashMap donde guardaremos los certificados que vamos a devolver.
		// Lo usaremos para guardar certificados usando como clave su Common Name.
		// Así, podremos detectar los Common Name repetidos y comparar las fechas
		// de emisión.
		if (SigDebug.isActive())
			SigDebug.write(providerName + " Filtrando certificados con Common Name duplicado...");
		
		HashMap map = new HashMap();
				
		int vectorSize = parsedCerts.size();
		for (int i = 0; i < vectorSize; i++) {
			
			ParsedCertificateImpl parsed = (ParsedCertificateImpl)parsedCerts.elementAt(i);
			
			if (SigDebug.isActive()) {
				SigDebug.write(providerName + " Alias: " + parsed.getAlias());
				SigDebug.write(providerName + " Emisión: " + parsed.getValidSince().getTime());
			}
			
			// Si el elemento no está en el mapa, lo añadimos
			if ( !map.containsKey(parsed.getCommonName()) ) {
				
				map.put(parsed.getCommonName(), parsed);
				
				if (SigDebug.isActive())
					SigDebug.write(providerName + " Almacenado certificado con alias " + parsed.getAlias() + " " +
						"(aún no existía ninguna entrada para el Common Name '" + parsed.getCommonName() + "')");
			
			// Si ya está en el mapa, comparamos fechas de emisión de los certificados.
			// Actualizamos elemento en el mapa si hace falta.
			} else {
				
				ParsedCertificateImpl storedParsed = (ParsedCertificateImpl)map.get(parsed.getCommonName());
				
				if (SigDebug.isActive()) {
					SigDebug.write(providerName + " Stored Alias: " + storedParsed.getAlias());
					SigDebug.write(providerName + " Stored Emisión: " + storedParsed.getValidSince().getTime());
				}
					
				if (parsed.getValidSince().getTime() > storedParsed.getValidSince().getTime()) {
					
					if (SigDebug.isActive()) {
						SigDebug.write(providerName + " " + parsed.getAlias() + " tiene una fecha de emisión más reciente que " + storedParsed.getAlias());
						SigDebug.write(providerName + " Actualizando certificado con Common Name '" + parsed.getCommonName() + "'");
					}
						
					map.put(parsed.getCommonName(), parsed);
				
				} else {
					
					if (SigDebug.isActive()) {
						SigDebug.write(providerName + " " + parsed.getAlias() + " tiene una fecha de emisión menor o igual que " + storedParsed.getAlias());
						SigDebug.write(providerName + " DESCARTADO...");
					}
					
				}				
				
			}
			
		}
		
		// Regeneramos vector con los certificados válidos.
		parsedCerts.clear();
		
		Set keySet = map.keySet();
		Iterator it = keySet.iterator();
		
		while ( it.hasNext() ) {
			
			String key = (String)it.next();
			ParsedCertificateImpl parsed = (ParsedCertificateImpl)map.get(key);
			parsedCerts.add(parsed);
			
		}
		
		if (SigDebug.isActive())
			SigDebug.write(providerName + " Filtrado completo.");
		
		return parsedCerts;
		
	}
	
	/**
	 * 
	 * @param alias
	 * @return
	 * @throws IOException 
	 * @throws CertificateEncodingException 
	 */
	protected ParsedCertificateImpl[] getParsedCertificates(String[] aliases) throws SignatureCertNotFoundException, SignaturePrivKeyException {
		
		Vector out = new Vector();
		
		for (int i = 0; i < aliases.length; i++) {
			
			X509Certificate[] chain;
			ParsedCertificateImpl parsed;
			
			try {
			
				chain = getCertChainFromAlias(aliases[i]);
				
				if (chain != null && chain.length > 0) {
				
					// Procesamos el elemento sólo si éste contiene cadena de certificación
					parsed = new ParsedCertificateImpl(chain, isInSecureDevice(aliases[i]), aliases[i]);
					out.add( parsed );
					
				}
			
			} catch (Exception e) {
			
				e.printStackTrace();
				throw new SignatureCertNotFoundException(e);
			
			}
						
		}
		
		return (ParsedCertificateImpl[])out.toArray(new ParsedCertificateImpl[out.size()]);
		
	}
	
	/**
	 * Comprueba si se considera que el certificado está en un dispositivo seguro o no.
	 * Toca implementarlo en el provider que extienda de esta clase.
	 * 
	 * @param alias
	 * @return
	 */
	protected abstract boolean isInSecureDevice(String alias);
	
	/**
	 * Nos devuelve la cadena de certificación de un alias específico.
	 * Toca implementarlo en el provider que extienda de esta clase.
	 * @param alias
	 * @return cadena de certificación de un alias específico.
	 * @throws SignatureProviderException
	 */
	protected abstract X509Certificate[] getCertChainFromAlias(String alias) throws SignatureProviderException;

	/**
	 * Nos devuelve una lista de los alias disponibles para el provider.
	 * Toca implementarlo en el provider que extienda de esta clase.
	 * @return
	 * @throws SignatureProviderException
	 */
	protected abstract String[] getAliases() throws SignatureProviderException;
	
	/**
	 * Código necesario para inicialización del provider (acceso al keystore, cargar DLLs, etc).
	 * Toca implementarlo en el provider que extienda de esta clase.
	 * @throws SignatureProviderException
	 */
	protected abstract void initialize() throws SignatureProviderException;
	
	/**
	 * @return the CAIB provider Name
	 */
	protected String getProviderName() {
		return providerName;
	}

	/**
	 * @param providerName the CAIB provider Name to set
	 */
	protected void setProviderName(String providerName) {
		this.providerName = providerName;
	}
	
	/***************************************************************************************************
	 * La implementación de los métodos de la interface es.caib.signatura.impl.SignerProviderInterface *
	 * es propia de cada provider, excepto getCertList(), por ahora...                                 *
	 ***************************************************************************************************/


	public ParsedCertificate[] getCertList(boolean recognized) throws SignatureCertNotFoundException, SignaturePrivKeyException  {
		
		String aliases[];
		
		try {
			
			aliases = getAliases();
			
		} catch (SignatureProviderException e) {
			
			throw new SignatureCertNotFoundException(e.getMessage(), e.getCause());
			
		}
		
		ParsedCertificateImpl[] parsedCerts = getParsedCertificates(aliases);
		ParsedCertificateImpl[] filteredCerts = filter(parsedCerts, recognized);
		
		return filteredCerts;
		
	}
	

	public abstract Date getCurrentDate(String certificateName, String password,
			boolean recognized) throws SignatureTimestampException,
			SignatureException, IOException;


	public abstract String getVersion();


	public abstract Signature sign(InputStream contentStream, String certificateName,
			String password, String contentType, boolean recognized,
			boolean timeStamp, boolean rawSign) throws IOException,
			SignatureException;


	public abstract void signPDF(InputStream contentStream, OutputStream signedStream,
			String certificateName, String password, String contentType,
			boolean recognized, String url, int position,
			boolean allowMultipleSignature) throws IOException,
			SignatureException;
	
	public abstract void signPDF(InputStream pdfInputStream, OutputStream signedStream,
			String certificateName,String password, String contentType, 
			boolean recognized, String textoAdicional, int stampOptions,
	        float top, float left,float height, float width, float rotation,
	        boolean allowMultipleSignature) throws IOException, SignatureException;

	public abstract void certifyDigitalCopy(InputStream contentStream,
			OutputStream signedStream, String certificateName, String password,
			String contentType, boolean recognized, String url,
			String localidad, float x, float y, float rotation, SignaturaProperties properties)
			throws IOException, SignatureException;
	
}
