/*
 * Created on 01/04/2005
 *
 */
package es.caib.signatura.provider.impl.common;

import java.io.IOException;
import java.security.cert.CertificateEncodingException;
import java.security.cert.CertificateExpiredException;
import java.security.cert.CertificateNotYetValidException;
import java.security.cert.X509Certificate;
import java.util.Date;
import java.util.Enumeration;
import java.util.Vector;

import org.bouncycastle.asn1.ASN1InputStream;
import org.bouncycastle.asn1.ASN1Sequence;
import org.bouncycastle.asn1.DERObject;
import org.bouncycastle.asn1.DERObjectIdentifier;
import org.bouncycastle.asn1.x509.PolicyInformation;
import org.bouncycastle.asn1.x509.TBSCertificateStructure;
import org.bouncycastle.asn1.x509.X509CertificateStructure;
import org.bouncycastle.asn1.x509.X509Extension;
import org.bouncycastle.asn1.x509.X509Extensions;
import org.bouncycastle.asn1.x509.X509Name;

import es.caib.signatura.impl.SigDebug;
import es.caib.signatura.impl.SignaturaProperties;


/**
 * @author u07286
 * 
 * Modificado por u91940 
 *
 * Clase encargada de procesar el certificado.
 * Establece si el certificado es reconocido o avanzado en función de las propiedades de la API
 * En caso que la CA del certificado no esté dada de alta en las propedades del API, se tratará como un certificado de pruebas, y se pondrá delante del nombre el literal "TEST: "
 * Delega la interpretación del certificado para que este cumpla con la interfaz ParsedCertificate a clases cargadas dinámicamente en función de la política del certificado, y que se configuran en las propiedades del API
 * En caso que en las propiedades de la PAI no aparezca ningún intérprete como delegado de la interpretación de el certificado, se usa el intérprete DefaultParsedCertificateImpl
 */

public class ParsedCertificateImpl implements  InternalParsedCertificate {
	InternalParsedCertificate impl=null;
	
	//propias del API
	private boolean recognized=false;
	private boolean test=false;
	private boolean unrecognized=false;
	private boolean securedevice=false;
	private int isInValidPeriod=0;
	
	public static final int Certificate_Not_Yet_Valid = -1;
	public static final int Certificate_Expired = 1;
	
	private SignaturaProperties properties = new SignaturaProperties ();

	private String alias;
		
	/**
	 * Guarda en impl el parser asignado a la política
	 */
	public ParsedCertificateImpl (X509Certificate cert[], boolean securedevice ) throws CertificateEncodingException, IOException
	{
		
		try {
			
			cert[0].checkValidity();
			
		} catch (CertificateNotYetValidException e) {
			
			isInValidPeriod = Certificate_Not_Yet_Valid;
		
		} catch (CertificateExpiredException e) {
			
			isInValidPeriod = Certificate_Expired;
		
		}
		
		this.securedevice=securedevice;
		
		byte b [] = cert[0].getEncoded ();
		ASN1InputStream asn1is = new ASN1InputStream ( b );
		DERObject obj = asn1is.readObject ();
		X509CertificateStructure certificate = new X509CertificateStructure ((ASN1Sequence) obj);
		TBSCertificateStructure tbs = certificate.getTBSCertificate ();
		X509Extensions exts = tbs.getExtensions ();
		Vector policies=new Vector();
		
		if (exts != null) {

			X509Extension polext = exts
					.getExtension(X509Extensions.CertificatePolicies);

			if (polext != null) {
				ASN1Sequence asn1seq;
				asn1is = new ASN1InputStream(polext.getValue().getOctets());
				asn1seq = (ASN1Sequence) asn1is.readObject();
				Enumeration tempPolicies = (asn1seq.getObjects());
				while (tempPolicies.hasMoreElements()) {
					ASN1Sequence seq2 = (ASN1Sequence) tempPolicies
							.nextElement();
					PolicyInformation info = new PolicyInformation(seq2);
					policies.add(info);
				}

				int cnt = 0;
				while (cnt < policies.size()) {
					PolicyInformation info = (PolicyInformation) policies
							.get(cnt);
					try {

						String parserClassName = properties
								.getCertificateParser("certificateparser."
										+ info.getPolicyIdentifier().getId());

						if (parserClassName != null) {
							Class parserClass = this.getClass()
									.getClassLoader()
									.loadClass(parserClassName);

							InternalParsedCertificate parser = (InternalParsedCertificate) parserClass
									.getConstructor(
											new Class[] {
													X509Certificate[].class,
													boolean.class })
									.newInstance(
											new Object[] { cert,
													new Boolean(securedevice) });
							if (parser != null) {
								impl = parser;
								break;
							}
						}
					} catch (Exception e) {
						e.printStackTrace();
					}
					cnt++;
				}
			}
		}
		if (impl == null) {
			impl = new DefaultParsedCertificateImpl(cert, securedevice);
		}
			
		//per ultim, processem la política segons el criteri definit en el signatura_api.properties
		processPolicy(cert,certificate,policies,exts);
		
		if( SigDebug.isActive() )  
			SigDebug.write(this.toString());
		
		
	}

	/**
	 * 
	 * @param cert
	 * @param securedevice
	 * @throws CertificateEncodingException
	 * @throws IOException
	 */
	public ParsedCertificateImpl(X509Certificate cert[], boolean securedevice, String alias) throws CertificateEncodingException, IOException
	{
		this(cert, securedevice);
		this.alias = alias;
	}

	private void processPolicy(	X509Certificate cert[],X509CertificateStructure certificate,Vector policies,X509Extensions extensions) throws CertificateEncodingException, IOException {
		recognized = false;
		unrecognized = false;
		test = false; 
		if( SigDebug.isActive() )  SigDebug.write( "ParsedCertificateImp.init: extensionss null ?" + ( extensions == null ? "si" : "no") );
		if( extensions != null ) {

			if (policies.size()!=0 ) {
				
				int cnt=0;
				while (cnt<policies.size()){
					PolicyInformation info = (PolicyInformation) policies.get(cnt);

					processPolicy (certificate,info.getPolicyIdentifier().getId(), extensions, securedevice);
					cnt++;
				}
			}else{
				processPolicy (certificate,"", extensions, securedevice);
				
			}
	
			// Determinar la CA
			X509Certificate caCert = cert[cert.length-1];
			String caName= getCACommonName(caCert);
			test = ! isIncluded(caName, properties.getRootCAs());
		}

		

	}
	
	private void processPolicy(X509CertificateStructure certificate,String id, X509Extensions exts, boolean securedevice) {

		if (isIncluded (id, properties.getRecognizedPolicies()))
			recognized = true;
		if (isIncluded (id, properties.getAdvancedPolicies()))
			unrecognized = true;
		if (isIncluded(id, properties.getAmbigousPoliciess()))
		{
			String recognizedExtensions[] = properties.getRecognizedExtensions();
			for (int i = 0; i < recognizedExtensions.length; i++)
			{
				X509Extension ext = exts.getExtension(new DERObjectIdentifier(recognizedExtensions[i]));
				if (ext != null)
					recognized = true;
				else
					unrecognized = true;
			}
		}
		if (!recognized && isIncluded(id, properties.getRecognizedSecureDevicePolicies()))
		{
			recognized = securedevice;
		}
		
		if("".equals(id) && "OU = FNMT Clase 2 CA,O = FNMT,C = ES".equals(certificate.getIssuer()))
			unrecognized = true; //PER ALS CERTIFICATS DE COMPONENT DE LA FNMT
	}
	

	private String getCACommonName (X509Certificate cert) throws IOException, CertificateEncodingException
	{
		// En cas de no trobar el CN, retornem el CN (posat pel cas de la FNMT, que no inclou
		// el CN en el certificat de l'emissor.
		String ou = "";
		byte b [] = cert.getEncoded ();
		ASN1InputStream asn1is = new ASN1InputStream ( b );
		DERObject obj = asn1is.readObject ();
		X509CertificateStructure certificate = new X509CertificateStructure ((ASN1Sequence) obj);
		asn1is.close();
		X509Name name = certificate.getIssuer();
		java.util.Vector v = name.getOIDs ();
		java.util.Vector value = name.getValues ();
		for ( int i = 0; i < v.size(); i++)
		{
			if (v.get(i).equals (X509Name.CN)){

				if(SigDebug.isActive())
					SigDebug.write("getCACommonName(): "+value.get(i).toString());
				return value.get(i).toString();
			}
			if (v.get(i).equals (X509Name.OU))
				ou = value.get(i).toString();
		}

		if(SigDebug.isActive())
			SigDebug.write("getCACommonName(): "+ou);

		// Si no troba el CN, retorna el OU.
		return ou;
	}

	private boolean isIncluded(String id, String[] policies) {
		for (int i = 0; i < policies.length; i++)
		{
			if (id.equals (policies[i]))
				return true;
		}
		return false;
	}

	/**
	 * Hasta la versión 2.5 devolvía el nombre del representante. 
	 * A partir de la versión 2.5 devuelve el nombre del Representado
	 */
	public String getName() {
		return impl.getName();
	}
	
	/* (non-Javadoc)
	 * @see es.caib.signatura.impl.CertificateParser#getNif()
	 */
	public String getNif ()
	{
		return impl.getNif();
	}
	
	
	/* (non-Javadoc)
	 * @see java.lang.Object#equals(java.lang.Object)
	 */
	public boolean equals(Object obj) {
		return impl.equals(obj);
	}
	/* (non-Javadoc)
	 * @see java.lang.Object#toString()
	 */
	public String toString() {
		String implDesc=impl.toString();
		
		if(SigDebug.isActive()){
			implDesc+= "\n Persona Física:"+impl.isPersonaFisica();			
			implDesc+= "\n Persona Jurídica:"+impl.isPersonaJuridica();
			implDesc+= "\n Reconocido: "+recognized;
			implDesc+= "\n Avanzado: "+unrecognized;
			implDesc+= "\n Parser: "+impl.getClass().getName();
			implDesc+= "\n common Name: "+getCommonName();
			implDesc+= "\n Name: "+getName();
			implDesc+= "\n full Name: "+getFullName();
			implDesc+= "\n";
		}
		
		return implDesc; 
		
	}
	/* (non-Javadoc)
	 * @see es.caib.signatura.impl.CertificateParser#getNifResponsable()
	 */
	public String getNifResponsable() {
		return impl.getNifResponsable();
	}
	/* (non-Javadoc)
	 * @see es.caib.signatura.impl.CertificateParser#isPersonaFisica()
	 */
	public boolean isPersonaFisica() {
		return impl.isPersonaFisica();
	}
	/* (non-Javadoc)
	 * @see es.caib.signatura.impl.CertificateParser#isPersonaJuridica()
	 */
	public boolean isPersonaJuridica() {
		return impl.isPersonaJuridica();
	}

	public boolean isRecognized ()
	{
		return recognized;
	}
	
	public boolean isAdvanced ()
	{
		return unrecognized;
	}

	public boolean isTest ()
	{
		return test;
	}
	
	
	
	public String getCommonName()
	{
		if (test)
			return "TEST: "+impl.getCommonName();
		else
			return impl.getCommonName();
	}

	/**
	 * Hasta la versión 2.5 devolvía el nombre del representante. 
	 * A partir de la versión 2.5 devuelve el nombre del Representado
	 */
	public String getFullName () {
		return impl.getFullName(); 
	}


	public Date getValidSince() {
		return impl.getValidSince();
	}


	public Date getValidUntil() {
		return impl.getValidUntil();
	}

	public String getAlias() {
		return alias;
	}
	
	/**
	 * @return the isInValidPeriod
	 */
	public synchronized int getIsInValidPeriod() {
		return isInValidPeriod;
	}
	
} 
