package es.caib.signatura.provider.impl.common;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.security.Provider;
import java.security.Security;
import java.security.cert.CRLException;
import java.security.cert.CertStore;
import java.security.cert.CertStoreException;
import java.security.cert.CertificateEncodingException;
import java.security.cert.X509CRL;
import java.security.cert.X509Certificate;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import org.bouncycastle.asn1.ASN1EncodableVector;
import org.bouncycastle.asn1.ASN1InputStream;
import org.bouncycastle.asn1.ASN1Object;
import org.bouncycastle.asn1.ASN1OctetString;
import org.bouncycastle.asn1.ASN1Set;
import org.bouncycastle.asn1.BERConstructedOctetString;
import org.bouncycastle.asn1.BERSet;
import org.bouncycastle.asn1.DEREncodable;
import org.bouncycastle.asn1.DERNull;
import org.bouncycastle.asn1.DERObject;
import org.bouncycastle.asn1.DERObjectIdentifier;
import org.bouncycastle.asn1.DERSet;
import org.bouncycastle.asn1.cms.CMSObjectIdentifiers;
import org.bouncycastle.asn1.cms.ContentInfo;
import org.bouncycastle.asn1.cms.SignedData;
import org.bouncycastle.asn1.pkcs.PKCSObjectIdentifiers;
import org.bouncycastle.asn1.x509.AlgorithmIdentifier;
import org.bouncycastle.asn1.x509.CertificateList;
import org.bouncycastle.asn1.x509.X509CertificateStructure;
import org.bouncycastle.cms.CMSException;
import org.bouncycastle.cms.CMSProcessableByteArray;
import org.bouncycastle.cms.CMSSignedData;
import org.bouncycastle.cms.SignerInformation;
import org.bouncycastle.cms.SignerInformationStore;
import org.bouncycastle.jce.provider.BouncyCastleProvider;

import es.caib.signatura.impl.SMIMEPkcs7InputStream;

import es.caib.signatura.api.Signature;
import es.caib.signatura.impl.RawSignature;
import es.caib.signatura.impl.SMIMEGenerator;
import es.caib.signatura.impl.SMIMEInputStream;

/**
 * Implementaci�n del generador S/MIME.
 * 
 * @author 3digits
 *
 */
public class SMIMEGeneratorImpl implements SMIMEGenerator {
	
	/**
	 * A partir del documento y de la firma generada sobre el mismo, se devuelve un documento en formato S/MIME a trav�s de un InputStream.
	 * 
	 * @param document Documento que ha sido firmado.
	 * @param signature Firma del documento
	 * 
	 * @return Documento SMIME.
	 * 
	 * @throws IOException En caso de no encontrar el documento pasado por par�metro.
	 */
	public InputStream generateSMIME(InputStream document, Signature signature) throws IOException {
		InputStream in;
		if( signature instanceof RawSignature ) {
			byte [] pkcs7 = encapsulateData(signature.getPkcs7(), document);
			in = new SMIMEPkcs7InputStream(pkcs7);
		}
		else {
			in = new SMIMEInputStream (signature, document);
		}
		
		return in;
	}
	
	/**
	 * Devuelve la firma pkcs7 con los datos encapsulados dentro de la firma. Si el array de bytes <code>pkcs7</code> no �s una firma
	 * bien formada el m�todo devolver� null. No se lanzan algunas excepciones porque el m�todo es privado
	 * y se supone que la firma que se pasa por par�metro es correcta.
	 * 
	 * @param pkcs7 array de bytes que contiene la firma pkcs7 dettached.
	 * @param data InputStream de los datos que se quieren encapsular dentro de la firma.
	 * @return firma en formato pkcs7 attached. 
	 * 
	 * @throws IOException
	 */
	private static byte[] encapsulateData(byte[] pkcs7, InputStream data) throws IOException {
		ASN1EncodableVector  digestAlgs = new ASN1EncodableVector();
	    ASN1EncodableVector  signerInfos = new ASN1EncodableVector();
	    CMSSignedData signedData;
	    try {
	    	signedData = new CMSSignedData(pkcs7);
	    }
	    catch (CMSException e) {
	    	e.printStackTrace();
	    	return null;
	    }
	    
	    CertStore certsAndCrls;
	    try{
			if(Security.getProvider("BC") == null){
				Provider p = new BouncyCastleProvider();
				Security.addProvider(p);
			}
			certsAndCrls = signedData.getCertificatesAndCRLs("Collection", "BC");
		}catch(Exception e){
			e.printStackTrace();
			return null;
		}
	  
		ByteArrayOutputStream out = new ByteArrayOutputStream();
		byte[] b = new byte[4096];
		int l = data.read(b);
		while (l > 0) {
			out.write(b, 0, l);
		  	l = data.read(b);
		}
		CMSProcessableByteArray content = new CMSProcessableByteArray(out.toByteArray());
		
		SignerInformationStore sis = signedData.getSignerInfos();
		Iterator it = sis.getSigners().iterator();
		while (it.hasNext()) {
		    SignerInformation signer = (SignerInformation)it.next();
		    AlgorithmIdentifier digAlgId = makeAlgId(signer.getDigestAlgOID(), signer.getDigestAlgParams());                
	        digestAlgs.add(digAlgId);
	        signerInfos.add(signer.toSignerInfo());
		}
		
		ASN1Set certificates = null;
		List certs = new ArrayList();
		try {
			certs.addAll(getCertificatesFromStore(certsAndCrls));
		}
	    catch (CertStoreException e) {
	    	e.printStackTrace();
	    	return null;
	    }
	    catch (IOException e) {
	    	e.printStackTrace();
	    	return null;
	    } 
	    catch (CertificateEncodingException e) {
	    	e.printStackTrace();
	    	return null;
	    }
		if (certs.size() != 0) {
		    certificates = createBerSetFromList(certs);
		}
		
		ASN1Set certrevlist = null;
		List crls = new ArrayList();
		try {
			crls.addAll(getCRLsFromStore(certsAndCrls));
		}
	    catch (CertStoreException e) {
	    	e.printStackTrace();
	        return null;
	    }
	    catch (IOException e) {
	    	e.printStackTrace();
	        return null;
	    }
	    catch (CRLException e) {
	    	e.printStackTrace();
	        return null;
	    }
		if (crls.size() != 0) {
		    certrevlist = createBerSetFromList(crls);
		}
		  
		DERObjectIdentifier  contentTypeOID = CMSObjectIdentifiers.data;
		ByteArrayOutputStream   bOut = new ByteArrayOutputStream();
		try {
		    content.write(bOut);
		}
		catch (CMSException e) {
			e.printStackTrace();
		    return null;
		}
		ASN1OctetString octs = new BERConstructedOctetString(bOut.toByteArray());
		ContentInfo encInfo = new ContentInfo(contentTypeOID, octs);
		
		SignedData  sd = new SignedData(
		                           new DERSet(digestAlgs),
		                           encInfo, 
		                           certificates, 
		                           certrevlist, 
		                           new DERSet(signerInfos));
		
		ContentInfo contentInfo = new ContentInfo(PKCSObjectIdentifiers.signedData, sd);
		  
		CMSSignedData attachedSignature = new CMSSignedData(contentInfo);
		
		return attachedSignature.getEncoded();
	}
	
	private static DERObject makeObj(byte[] encoding) throws IOException {
        if (encoding == null) {
            return null;
        }

        ByteArrayInputStream    bIn = new ByteArrayInputStream(encoding);
        ASN1InputStream         aIn = new ASN1InputStream(bIn);

        return aIn.readObject();
    }
	  
	private static AlgorithmIdentifier makeAlgId(String oid, byte[] params) throws IOException {
	    if (params != null) {
	        return new AlgorithmIdentifier(
	                        new DERObjectIdentifier(oid), makeObj(params));
	    }
	    else {
	        return new AlgorithmIdentifier(
	                        new DERObjectIdentifier(oid), new DERNull());
	    }
	}
	
	private static ASN1Set createBerSetFromList(List derObjects) {
	     ASN1EncodableVector v = new ASN1EncodableVector();
	     
	     for (Iterator it = derObjects.iterator(); it.hasNext();) {
	        v.add((DEREncodable)it.next());
	     }
	     
	     return new BERSet(v);
	}
		
	private static List getCertificatesFromStore(CertStore certStore) throws CertStoreException, IOException, CertificateEncodingException{
	    List certs = new ArrayList();
	
        for (Iterator it = certStore.getCertificates(null).iterator(); it.hasNext();) {
            X509Certificate c = (X509Certificate) it.next();
            certs.add(X509CertificateStructure.getInstance(ASN1Object.fromByteArray(c.getEncoded())));
        }
        
        return certs;
	}
	
	private static List getCRLsFromStore(CertStore certStore) throws CertStoreException, IOException, CRLException {
	    List crls = new ArrayList();
	
        for (Iterator it = certStore.getCRLs(null).iterator(); it.hasNext();) {
            X509CRL c = (X509CRL) it.next();
            crls.add(CertificateList.getInstance(ASN1Object.fromByteArray(c.getEncoded())));
        }

        return crls;
	}
}