package es.caib.signatura.provider.impl.common;

import java.io.ByteArrayInputStream;
import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.math.BigInteger;
import java.net.HttpURLConnection;
import java.net.MalformedURLException;
import java.net.URL;
import java.net.URLConnection;
import java.security.NoSuchAlgorithmException;
import java.security.NoSuchProviderException;
import java.security.cert.X509Certificate;
import java.util.Collection;
import java.util.Date;
import java.util.Random;

import org.bouncycastle.asn1.ASN1InputStream;
import org.bouncycastle.asn1.DEREncodableVector;
import org.bouncycastle.asn1.DERInteger;
import org.bouncycastle.asn1.DERObject;
import org.bouncycastle.asn1.DERObjectIdentifier;
import org.bouncycastle.asn1.DEROctetString;
import org.bouncycastle.asn1.DERSequence;
import org.bouncycastle.asn1.DERSet;
import org.bouncycastle.asn1.cms.AttributeTable;
import org.bouncycastle.asn1.cms.Attribute;
import org.bouncycastle.cms.CMSException;
import org.bouncycastle.cms.CMSSignedData;
import org.bouncycastle.cms.SignerInformation;
import org.bouncycastle.cms.SignerInformationStore;
import org.bouncycastle.crypto.digests.SHA1Digest;
import org.bouncycastle.tsp.TSPAlgorithms;
import org.bouncycastle.tsp.TSPException;
import org.bouncycastle.tsp.TimeStampRequest;
import org.bouncycastle.tsp.TimeStampRequestGenerator;
import org.bouncycastle.tsp.TimeStampResponse;
import org.bouncycastle.tsp.TimeStampToken;
import org.bouncycastle.tsp.TimeStampTokenInfo;

import es.caib.signatura.afirma.rfc3161.AFirmaURLStreamHandlerFactoryImpl;
import es.caib.signatura.afirma.rfc3161.RFC3161Connection;
import es.caib.signatura.afirma.rfc3161.TsaOutputStream;
import es.caib.signatura.api.SignatureTimestampException;
import es.caib.signatura.impl.SigDebug;
import es.caib.signatura.impl.SignaturaProperties;
import es.caib.signatura.provider.impl.common.SHA1Util;

public class TimeStampManager {
	private SignaturaProperties properties = null;

	private static Random r = new Random (System.currentTimeMillis());
	
	private TimeStampToken lastTimeStampTokenGenerated = null;
	
	private static String OID_MAP_TSA2="1.3.4.6.1.3.4.6"; /** PJR 21-09-09 adaptación a los requisitos de @firma **/
	
	public  TimeStampToken getLastTimeStampTokenGenerated(){
		return lastTimeStampTokenGenerated;
	}
	
	private URL getURLPrincipal(X509Certificate cert )  
		throws IOException, SignatureTimestampException
	{
		if( SigDebug.isActive() )  {
			SigDebug.write( "Obtenint proveidor de segell de temps per " + cert.getIssuerX500Principal().getName() );
		}
		// Seleccionar servidor de temps segons certificat.
		String proveidor = properties.getTimestampService( cert.getIssuerX500Principal().getName() );
		if( proveidor == null || proveidor.length() == 0 ) {
			proveidor = properties.getTimestampService( "default" );
		}
		if( proveidor == null || proveidor.length() == 0 ) {
			throw new SignatureTimestampException( "No se ha configurado un servidor de sello de tiempo para " + cert.getIssuerX500Principal().getName() );
		}
		if( SigDebug.isActive() )  {
			SigDebug.write( "Trobat proveidor de segell de temps per " + cert.getIssuerX500Principal().getName() 
					+ ", és: " + proveidor );
		}
		
		/** creem la URL, i la configurem per a que faci servir la factoria de protocols privada **/
		URL url;
		try{
			url=new URL(proveidor);
		}catch(MalformedURLException urle){
			String protocol=proveidor.substring(0,proveidor.indexOf(":"));
			url=new URL(null,proveidor,new AFirmaURLStreamHandlerFactoryImpl().createURLStreamHandler(protocol));
		}
		
		return url;
	}

	private String getPrincipalApplicationId(X509Certificate cert )  throws IOException, SignatureTimestampException
	{
		if( SigDebug.isActive() )  {
			SigDebug.write( "Obtenint identificador applicationId per al segell de temps per " + cert.getIssuerX500Principal().getName() );
		}
		// Seleccionar servidor de temps segons certificat.
		String applicationId= properties.getTimestampServiceApplicationId( cert.getIssuerX500Principal().getName() ); /** PJR 21-09-09 adaptación a los requisitos de @firma **/
		
		if( applicationId == null ) {
			applicationId = properties.getTimestampServiceApplicationId( "default" ); /** PJR 21-09-09 adaptación a los requisitos de @firma **/
		}

		if( SigDebug.isActive() )  {
			SigDebug.write( "Trobat applicationId del proveidor de segell de temps per " + cert.getIssuerX500Principal().getName() 
					+ ", és: " + applicationId );
		}
		return applicationId;
	}
	
	private String getPrincipalPolicyId(X509Certificate cert )  throws IOException, SignatureTimestampException
	{
		if( SigDebug.isActive() )  {
			SigDebug.write( "Obtenint identificador policyOID per al segell de temps per " + cert.getIssuerX500Principal().getName() );
		}
		// Seleccionar servidor de temps segons certificat.
		String policyOID= properties.getTimestampServicePolicyOID( cert.getIssuerX500Principal().getName() ); /** PJR 21-09-09 adaptación a los requisitos de @firma **/
		
		if( policyOID == null ) {
			policyOID = properties.getTimestampServicePolicyOID( "default" ); /** PJR 21-09-09 adaptación a los requisitos de @firma **/
		}

		if( SigDebug.isActive() )  {
			SigDebug.write( "Trobat policyOID del proveidor de segell de temps per " + cert.getIssuerX500Principal().getName() 
					+ ", és: " + policyOID );
		}
		return policyOID;
	}	
	
	public TimeStampManager() {
		super();
		try {
			properties = new SignaturaProperties ();
		} catch( Exception e ) {
			throw new Error( "No se ha podido obtener la configuración de la API de firma digital.", e );
		}
	}

	/**
	 * @param cert: certificado del firmante
	 * @param digest: digest del documento original
	 * @param digestAlgorithm: algoritmo de digest
	 * @return
	 * @throws IOException
	 * @throws TSPException
	 * @throws SignatureTimestampException
	 */
	private TimeStampToken generateTimeStamp(X509Certificate cert, byte digest[], String digestAlgorithm) throws IOException, TSPException,
			SignatureTimestampException {
		try {
			// Generar la petición
			TimeStampRequestGenerator generator = new TimeStampRequestGenerator();
			generator.setCertReq(true);
			
			String applicationId=getPrincipalApplicationId(cert); /** PJR 21-09-09 adaptación a los requisitos de @firma **/
			if(applicationId!=null && !"".equals(applicationId)) 
				generator.addExtension(OID_MAP_TSA2, false, new DEROctetString(applicationId.getBytes())); 
			
			String policyOID=getPrincipalPolicyId(cert);
			if(policyOID!=null && !"".equals(policyOID))
				generator.setReqPolicy(policyOID);
			
			byte[] nonce = new byte[16];
			r.nextBytes(nonce);
			TimeStampRequest req = generator.generate(digestAlgorithm, digest, new BigInteger(nonce));
			 //java.io.FileOutputStream  a=new java.io.FileOutputStream("c:/util/ts_req_1.txt"); int i=0;for(i=0;i<encoded.length;i++){if(encoded[i]!=-1)a.write(encoded[i]);} ;a.close();

			// Enviar al servicio de sellado de tiempo..
			// Seleccionar servidor de temps segons certificat.
			URL url = getURLPrincipal(cert);

			URLConnection conn = (URLConnection) url.openConnection();
			if(conn instanceof HttpURLConnection){
				return getHTTPTimestamp(conn,req);
			}else if (conn instanceof RFC3161Connection){
				return getRFC3161TimeStamp((RFC3161Connection)conn,req);
	            
			}
		} catch (Exception e) {
			e.printStackTrace();
			throw new SignatureTimestampException(e);
		}
		return null;
	}
 
	private TimeStampToken getRFC3161TimeStamp(RFC3161Connection conn, TimeStampRequest req) throws IOException, SignatureTimestampException, TSPException, CMSException {
		TsaOutputStream dataoutputstream = conn.getTsaOutputStream();
        dataoutputstream.write(req.getEncoded());
        dataoutputstream.flush();
        
       

        DERSequence asn = (DERSequence)conn.getTsaInputStream().readObject();
        DERSequence info = (DERSequence)asn.getObjectAt(0);
        DERInteger status = (DERInteger)info.getObjectAt(0);

        dataoutputstream.close();
        conn.getTsaInputStream().close();
        
        if(status.getValue().intValue() != 0 && status.getValue().intValue() != 1)
            throw new SignatureTimestampException("No se ha podido obtener el sello de tiempo RFC3161: "+ status.getValue().intValue());

        
        return new TimeStampToken(new CMSSignedData(asn.getObjectAt(1).getDERObject().getDEREncoded()));

        
}

	private TimeStampToken getHTTPTimestamp(URLConnection conn,TimeStampRequest req) throws IOException, TSPException,SignatureTimestampException {
		//conn.setRequestProperty("Host", url.getHost() + ":" + url.getPort());
		conn.setRequestProperty("Content-Type", "application/timestamp-query");
		//conn.setRequestProperty("Content-Length", Integer.toString(encoded.length));
		((HttpURLConnection)conn).setRequestMethod("POST");
		conn.setUseCaches(false);
		conn.setDefaultUseCaches(false);
		conn.setDoInput(true);
		conn.setDoOutput(true);
		OutputStream out = conn.getOutputStream();
		out.write(req.getEncoded());
		out.flush();
		out.close();
		// Recibir la respuesta
		InputStream in = conn.getInputStream();  //java.io.FileOutputStream  a=new java.io.FileOutputStream("c:/util/ts_1.txt");int b; do {b=in.read();if(b!=-1)a.write(b);} while(b!=-1);a.close();
		TimeStampResponse response = new TimeStampResponse(in); //componerSalida(in) 
		in.close();

		//validamos la firma de la respuesta
		response.validate(req);

		//miramos que no haya habido errores en la petición
		if(response.getFailInfo()!=null){
			throw new SignatureTimestampException("No se ha podido obtener el sello de tiempo: PKIFailInfo:"+response.getFailInfo());
		}

		//devolvemos el token
		return response.getTimeStampToken();
		
	}

	/**
	 * @param cert: certificado del firmante
	 * @param signedData: datos firmados sin sello de tiempo
	 * @param digest: digest del documento original
	 * @param digestAlogrithm: algoritmo de digest
	 * @return: datos firmados con sello de tiempo si se consigue el sello de tiempo, sino devuelve nulo
	 * @throws IOException
	 * @throws TSPException
	 * @throws SignatureTimestampException
	 * 
	 */
	public CMSSignedData addTimestamp(X509Certificate cert, CMSSignedData signedData,
			byte digest[], String algorithm) 
		throws IOException, TSPException, SignatureTimestampException 
	{
		try{
			// Obtener los datos del firmate
			Collection ss = signedData.getSignerInfos().getSigners();
			SignerInformation si = (SignerInformation) ss.iterator().next();

			// Generar el time stamp
			TimeStampToken tok = generateTimeStamp(cert, digest, algorithm);

			// Leer el timestampo como un objeto DER
			ASN1InputStream asn1InputStream = new ASN1InputStream (tok.getEncoded());
			DERObject tstDER = asn1InputStream.readObject();
			// Empaquetar el timestamp dentro de  un set
			DERSet ds = new DERSet(tstDER);
			// Crear el atributo formado por el OID asociado a un timestamp 
			Attribute a = new Attribute(new DERObjectIdentifier("1.2.840.113549.1.9.16.2.14"), ds);

			// Crear el vector de atributos. Como no había añadido ninguno, parto de un vector vacío
			DEREncodableVector dv = new DEREncodableVector();
			dv.add(a);
			// Construyo la tablad e Atributos sin firmar
			AttributeTable at = new AttributeTable(dv);
			si = SignerInformation.replaceUnsignedAttributes(si, at);
			// Borro la lista de firmantes y la sustituyo por el nuevo firmato
			ss.clear();
			ss.add(si);
			SignerInformationStore sis = new SignerInformationStore(ss);
			signedData = CMSSignedData.replaceSigners(signedData, sis);

			this.lastTimeStampTokenGenerated = tok;
		}catch(Exception e){
			/*
			 * Se devuelve nulo para hacer notar que se ha producido un fallo 
			 */
			return null;
		}		
		return signedData;
	}

	/**
	 * @param cert: certificado del firmante
	 * @param signedData: datos firmados sin sello de tiempo
	 * @return: datos firmados con sello de tiempo si se consigue el sello de tiempo, sino devuelve nulo
	 * @throws IOException
	 * @throws TSPException
	 * @throws SignatureTimestampException
	 */
	public CMSSignedData addTimestamp(X509Certificate cert, CMSSignedData signedData) 
		throws IOException, TSPException, SignatureTimestampException 
	{
		// Obtener los datos del firmate
		Collection ss = signedData.getSignerInfos().getSigners();
		SignerInformation si = (SignerInformation) ss.iterator().next();

		// Obtener la firma
		byte digest[];
		try {
			digest = SHA1Util.digest(si.getSignature());
		} catch (NoSuchAlgorithmException e) {
			throw new SignatureTimestampException(e);
		} catch (NoSuchProviderException e) {
			throw new SignatureTimestampException(e);
		}
		// Generar el time stamp
		return addTimestamp (cert, signedData, digest, TSPAlgorithms.SHA1);
	
	}
	
	public CMSSignedData addWrongTimestamp(X509Certificate cert, CMSSignedData signedData, byte digest[], String digestAlogrithm)
			throws IOException, TSPException, SignatureTimestampException {
		
		// Generar el time stamp
		//TimeStampToken tok = generateTimeStamp(cert, digest, digestAlogrithm);

		// Obtener los datos del firmante
		//Collection ss = signedData.getSignerInfos().getSigners();
		//SignerInformation si = (SignerInformation) ss.iterator().next();

		//byte data[] = new byte[0];
		// Leer el timestamp como un objeto DER
		//ASN1InputStream asn1InputStream = new ASN1InputStream(data);
		//DERObject tstDER = asn1InputStream.readObject();
		// Empaquetar el timestamp dentro de un set
		//DERSet ds = new DERSet(tstDER);
		// Crear el atributo formado por el OID asociado a un timestamp
		//Attribute a = new Attribute(new DERObjectIdentifier("1.2.840.113549.1.9.16.2.14"), ds);

		// Crear el vector de atributos. Como no había añadido ninguno, parto de
		// un vector vacío
		//DEREncodableVector dv = new DEREncodableVector();
		//dv.add(a);
		// Construyo la tablad e Atributos sin firmar
		//AttributeTable at = new AttributeTable(dv);
		//si = SignerInformation.replaceUnsignedAttributes(si, at);
		// Borro la lista de firmantes y la sustituyo por el nuevo formato
		//ss.clear();
		//ss.add(si);
		//SignerInformationStore sis = new SignerInformationStore(ss);
		//signedData = CMSSignedData.replaceSigners(signedData, sis);
		
		return signedData;
	}
	
	public Date getTimeStamp (X509Certificate cert) throws IOException, TSPException, SignatureTimestampException
	{
		// Generar la petición
		TimeStampRequestGenerator generator = new TimeStampRequestGenerator ();
		generator.setCertReq(true);
//				generator.setReqPolicy("1.3.6.1.4.1.17326.10.9.2");
		byte [] nonce = new byte[16];
		r.nextBytes(nonce);
		byte [] emptyBuffer = new byte [new SHA1Digest().getDigestSize()];
		TimeStampRequest req = generator.generate(TSPAlgorithms.SHA1, emptyBuffer, 
				new BigInteger(nonce));
		byte encoded []= req.getEncoded();
		
		// Enviar a CAMERFIRMA
		// URL url = new URL("http://tsa.camerfirma.com:15004/TSUSoftware");
		URL url = getURLPrincipal( cert );
		URLConnection conn =  url.openConnection();
		// conn.setRequestProperty("Host", "tsa.camerfirma.com:15004");
		conn.setRequestProperty( "Host", url.getHost() + ":" + url.getPort() );
		conn.setRequestProperty("Content-type", "application/timestamp-query");
		conn.setRequestProperty("Content-length", Integer.toString ( encoded.length ) );
		conn.setDoInput(true);
		conn.setDoOutput(true);
		OutputStream out = conn.getOutputStream();
		out.write(encoded);
		out.flush();
		out.close();
		// Recibir la respuesta
		InputStream in = conn.getInputStream();
		TimeStampResponse response = new TimeStampResponse (in);
		in.close ();
		response.validate(req);
		return response.getTimeStampToken().getTimeStampInfo().getGenTime();
	}

}
