package es.caib.signatura.api;

import java.io.IOException;
import java.io.InputStream;
import java.security.cert.X509Certificate;
import java.util.Date;


/**
 * Interface to acces to the digital signature of a document. It provides the necessary methods to get
 * the digital signature and information associated to the certificate used to sign the document.
 * 
 * @author Jesús Reyes (3dígits)
 * @see Signature
 * 
 */

public interface Signature {

	/**
	 * Gets the name of the certification authority of the certificate used to sign.
	 * 
	 * @return the name of the certification authority.
	 */
	public String getCertCaName();

	/**
	 * Gets the subject's Common Name of the certificate used to sign.
	 * 
	 * @return the subject common name.
	 */
	public String getCertSubjectCommonName();

	/**
	 * Gets the concatenation of the subject's alternate name of the certificate used to sign
	 * as follows: name0 = value, name1 = value, ...
	 * 
	 * @return a string containing the subject's alternate name of the certificate.
	 */	
	public String getCertSubjectAlternativeNames();


	/**
	 * Gets the date of the signature timestamp.
	 * 
	 * @return the date of the signature timestamp or <code>null</code> if the signature doesn't include timestamp.
	 */
	public Date getDate() throws SignatureTimestampException;

	/**
	 * Determinate the certificate validity of the signature.
	 * 
	 * @see Signer#verify
	 * @return <code>true</code> if the certificate is valid; <code>false</code> otherwise.
	 */
	public boolean verify() throws SignatureVerifyException;

	/**
	 * Returns the signature in PKCS#7 format.
	 * 
	 * @return a byte array containing the signature in PKCS#7 format.
	 */
	public byte[] getPkcs7();

	/**
	 * Returns the content type of the certificate.
	 * 
	 * @return the content type of the certificate.
	 */
	public String getContentType();
  
	/**
	 * Returns the X509 certificate used to sign.
	 * 
	 * @return the X509Certificate used to sign.
	 */  
	public X509Certificate getCert();
  
	/**
	 * Returns the certificate chain.
	 * 
	 * @return the certificate chain.
	 */
	public X509Certificate[] getCertificateChain() throws Exception;
  
	/**
	 * Devuelve el Seycon Principal a partir del certificado con el que se ha firmado.
	 * @return SeyconPrincipal
	 */  
	public ParsedCertificate getParsedCertificate();

	/**
	 * Verifies the digital signature of a document. The verification process is independent of signature timestamp.
	 * 
	 * @param contentStream byte stream of the document.
	 * @return <code>true</code> if the verification process is correct; <code>false</code> otherwise.
	 * @throws SignatureProviderException If the API provider cannot be accessed.
	 * @throws IOException  If the document or the timestamp server is not available.
	 * @throws SignatureVerifyException If failed the verification process.
	 */
	public boolean verify(InputStream contentStream)
		throws SignatureProviderException, IOException, SignatureVerifyException;

	/**
	 * Verifies the digital signature of a document. If the digital signature requires a timestamp
	 * and don't have it then a timestamp is added.

	 * Si la firma requiere sello de tiempo y no dispone de él, se intenta añadir el sello de tiempo
	 * 
	 * @param contentStream byte stream of the document.
	 * @return <code>true</code> if the verification process is correct; <code>false</code> otherwise.
	 * @throws SignatureProviderException If the API provider cannot be accessed.
	 * @throws IOException  If the document or the timestamp server is not available.
	 * @throws SignatureVerifyException If failed the verification process.
	 */
	public boolean verifyAPosterioriTimestamp(InputStream contentStream)
		throws SignatureProviderException, IOException, SignatureVerifyException;
	
}