
package es.caib.signatura.api;


import java.io.InputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.OutputStream;
import java.security.UnrecoverableKeyException;
import java.security.cert.X509Certificate;
import java.util.Date;
import java.net.*;

/**Interfaz que independiza las implementaciones de firma y verificación de cada entidad certificadora con una
 * serie de métodos estándar, llevando a cabo la firma digital desacoplada (firma y documento original
 * se mantienen por separado). Esta firma puede ser avanzada o avanzada con sello de tiempo.
 * La firma se encapsula con la clase <code>Signature</code>.
 * 
 * @author Jesús Reyes (3dígits)
 * @version 1.0
 * @see Signature
 */
public interface Signer {

	/**
	 * Indica que la firma se añadirá en la parte superior del PDF.
	 */
	static final int PDF_SIGN_POSITION_TOP = 1;
	
	/**
	 * Indica que la firma se añadirá en la parte inferior del PDF.
	 */
	static final int PDF_SIGN_POSITION_BOTTOM = 2;
	
	/**
	 * Indica que la firma se añadirá en el lado izquierdo del PDF.
	 */
	static final int PDF_SIGN_POSITION_LEFT = 4;
	
	/**
	 * Indica que la firma se añadirá en el lado derecho del PDF.
	 */
	static final int PDF_SIGN_POSITION_RIGHT = 8;

	/**
	 * Indica que no se añadirá ni código de barras ni información del firmante.
	 */
	static final int PDF_SIGN_POSITION_NONE = 0;

	static final int PDF_SIGN_PAGE_LAST = 16;
	static final int PDF_SIGN_DEFAULT_SIGNATURE_APPERANCE = 32;


/**
 * Obtiene la lista de certificados disponibles en el almacén de certificados definido por la API 
 * de la entidad certificadora: disco duro, dispositivo USB, etc.
 * @return lista de los nombres de certificado disponibles
 * @throws SignatureCertNotFoundException  si no se encuentra ningún certificado disponible
 * @throws SignaturePrivKeyException  si no se pueden obtener las claves privadas de los certificados
 */
	public  String[] getCertList(String contentType) throws SignatureCertNotFoundException, SignaturePrivKeyException;

/**
 * Obtiene la firma digital de un documento de disco y la encapsula en un objeto
 * <code>Signature</code> sin sello de tiempo.
 * @param fileName nombre del fichero que se desea firmar
 * @param certificateName nombre del certificado que se usará para firmar
 * @param password contraseña de la clave privada del usuario
 * @param contentType tipo MIME del documento a firmar
 * @return {@link Signature} firma del documento
 * @throws FileNotFoundException  si no se encuentra el fichero a firmar
 * @throws IOException  si ha habido algún problema al abrir el fichero
 * @throws SignatureProviderException  si no se ha podido acceder a la API del proveedor de firma electrónica
 * @throws SignatureSignException  si ha habido algún problema en el proceso de firma
 * @throws UnrecoverableKeyException  si la contraseña de la clave privada no es correcta
 * @throws SignatureException
 */
	public  Signature sign (String fileName, String certificateName, String password, String contentType) 
          throws IOException, SignatureException;

/**
 * Obtiene la firma digital de un documento pasado como <code>InputStream</code> y la encapsula en un objeto
 * <code>Signature</code> sin sello de tiempo.
 * @param contentStream flujo de bytes del documento a firmar
 * @param certificateName nombre del certificado que se usará para firmar
 * @param password contraseña de la clave privada del usuario
 * @param contentType tipo MIME del documento a firmar
 * @return {@link Signature} firma del documento
 * @throws IOException  si ha habido algún problema de comunicación al pasar el stream de bytes
 * del documento
 * @throws SignatureProviderException  si no se ha podido acceder a la API del proveedor de firma electrónica
 * @throws SignatureSignException  si ha habido algún problema en el proceso de firma
 * @throws UnrecoverableKeyException  si la contraseña de la clave privada no es correcta
 * @throws SignatureException
 */

  public  Signature sign (InputStream contentStream, String certificateName, String password, String contentType) 
          throws IOException, SignatureException;

  /**
   * Firma digitalmente de un documento PDF pasado como <code>InputStream</code> y devuelve el mismo PDF firmado
   * y modificado de forma que, en uno de los bordes del documento que se le indique, aparezca el firmante, 
   * la url desde la que se puede consultar el PDF y una matriz de puntos en formato PDF417 que continene 
   * esa misma URL. 
   * @param contentStream flujo de bytes del documento PDF a firmar
   * @param certificateName nombre del certificado que se usará para firmar
   * @param password contraseña de la clave privada del usuario
   * @param contentType tipo MIME del documento a firmar
   * @param url URL que se sobreimpresiona sobre el documento firmado
   * @param position Posición en la que se se sobreimpresionará la url en cada una de las hojas del documento firmado. Pueden ser PDF_SIGN_POSITION_TOP,
   * PDF_SIGN_POSITION_BOTTOM, PDF_SIGN_POSITION_RIGHT o PDF_SIGN_POSITION_LEFT 
   * @return documento PDF firmado
   * @throws IOException  si ha habido algún problema de comunicaciÃ³n al pasar el stream de bytes
   * del documento
   * @throws SignatureProviderException  si no se ha podido acceder a la API del proveedor de firma electrónica
   * @throws SignatureSignException  si ha habido algún problema en el proceso de firma
   * @throws UnrecoverableKeyException  si la contraseña de la clave privada no es correcta
   * @throws SignatureException
   */

    public void signPDF (InputStream contentStream, OutputStream signedStream, String certificateName, String password, String contentType, String url, int position) 
            throws IOException, SignatureException;
  
  /**
   * Firma digitalmente de un documento PDF pasado como <code>InputStream</code> y devuelve el mismo PDF firmado
   * y modificado de forma que, en uno de los bordes del documento que se le indique, aparezca el firmante, 
   * la url desde la que se puede consultar el PDF y una matriz de puntos en formato PDF417 que continene 
   * esa misma URL. 
   * @param contentStream flujo de bytes del documento PDF a firmar
   * @param certificateName nombre del certificado que se usará para firmar
   * @param password contraseña de la clave privada del usuario
   * @param contentType tipo MIME del documento a firmar
   * @param url URL que se sobreimpresiona sobre el documento firmado
   * @param position Posición en la que se se sobreimpresionará la url en cada una de las hojas del documento firmado. Pueden ser PDF_SIGN_POSITION_TOP,
   * PDF_SIGN_POSITION_BOTTOM, PDF_SIGN_POSITION_RIGHT o PDF_SIGN_POSITION_LEFT 
   * @return documento PDF firmado
   * @throws IOException  si ha habido algún problema de comunicaciÃ³n al pasar el stream de bytes
   * del documento
   * @throws SignatureProviderException  si no se ha podido acceder a la API del proveedor de firma electrónica
   * @throws SignatureSignException  si ha habido algún problema en el proceso de firma
   * @throws UnrecoverableKeyException  si la contraseña de la clave privada no es correcta
   * @throws SignatureException
   * 
   * @deprecated
   */

    public OutputStream signPDF (InputStream contentStream, String certificateName, String password, String contentType, String url, int position) 
            throws IOException, SignatureException;
  
/**
 * Obtiene la firma digital de un documento de una URL y la encapsula en un objeto
 * <code>Signature</code> sin sello de tiempo.
 * @param url dirección URL del fichero que se desea firmar
 * @param certificateName nombre del certificado que se usará para firmar
 * @param password contraseña de la clave privada del usuario
 * @param contentType tipo MIME del documento a firmar
 * @return {@link Signature} firma del documento
 * @throws FileNotFoundException  si no se encuentra el fichero a firmar
 * @throws IOException  si ha habido algún problema al abrir el fichero
 * @throws SignatureProviderException  si no se ha podido acceder a la API del proveedor de firma electrónica
 * @throws SignatureSignException  si ha habido algún problema en el proceso de firma
 * @throws UnrecoverableKeyException  si la contraseña de la clave privada no es correcta
 * @throws SignatureException
 */
	public  Signature sign (URL url, String certificateName, String password, String contentType) 
          throws IOException, SignatureException;

/**
 * Verifica la firma digital de un documento pasado como stream de bytes a partir de la firma encapsulada
 * en un objeto <code>Signature</code>
 * Si la firma requiere sello de tiempo y no dispone de él, se intenta añadir el sello de tiempo
 * @param contentStream flujo de bytes del documento original
 * @param signatureData objeto que contiene la firma
 * @return <code>true</code> si la verificación es correcta y <code>false</code> en caso contrario
 * @throws SignatureProviderException  si no se ha podido acceder a la API del proveedor de firma electrónica
 * @throws IOException  si no se ha podido acceder al fichero o si no se ha podido contactar con
 * el servidor de sello de tiempo
 * @throws SignatureVerifyException  si no se ha podido realizar el proceso de verificación
 */
	public  boolean verifyAPosterioriTimeStamp(InputStream contentStream, Signature signatureData)
          throws SignatureProviderException, IOException, SignatureVerifyException;
			
/**
 * Verifica la firma digital de un documento pasado como stream de bytes a partir de la firma encapsulada
 * en un objeto <code>Signature</code>
 * Si la firma requiere sello de tiempo y no dispone de él, se intenta añadir el sello de tiempo
 * @param url del documento original
 * @param signatureData objeto que contiene la firma
 * @return <code>true</code> si la verificación es correcta y <code>false</code> en caso contrario
 * @throws SignatureProviderException  si no se ha podido acceder a la API del proveedor de firma electrónica
 * @throws IOException  si no se ha podido acceder al fichero o si no se ha podido contactar con
 * el servidor de sello de tiempo
 * @throws SignatureVerifyException  si no se ha podido realizar el proceso de verificación
 */
	
	public  boolean verifyAPosterioriTimeStamp(URL url, Signature signatureData)
          throws SignatureProviderException, IOException, SignatureVerifyException;
			
/**
 * Verifica la firma digital de un documento pasado como stream de bytes a partir de la firma encapsulada
 * en un objeto <code>Signature</code>
 * Si la firma requiere sello de tiempo y no dispone de él, se intenta añadir el sello de tiempo
 * @param nombre del fichero del documento original
 * @param signatureData objeto que contiene la firma
 * @return <code>true</code> si la verificación es correcta y <code>false</code> en caso contrario
 * @throws SignatureProviderException  si no se ha podido acceder a la API del proveedor de firma electrónica
 * @throws IOException  si no se ha podido acceder al fichero o si no se ha podido contactar con
 * el servidor de sello de tiempo
 * @throws SignatureVerifyException  si no se ha podido realizar el proceso de verificación
 */
	public  boolean verifyAPosterioriTimeStamp(String fileName, Signature signatureData)
          throws SignatureProviderException, IOException, SignatureVerifyException;
			
/**
 * Verifica la firma digital de un documento pasado como stream de bytes a partir de la firma encapsulada
 * en un objeto <code>Signature</code>
 * La verificación es independiente de si la firma llevaba o no sello de tiempo
 * @param contentStream flujo de bytes del documento original
 * @param signatureData objeto que contiene la firma
 * @return <code>true</code> si la verificación es correcta y <code>false</code> en caso contrario
 * @throws SignatureProviderException  si no se ha podido acceder a la API del proveedor de firma electrónica
 * @throws IOException  si no se ha podido acceder al fichero o si no se ha podido contactar con
 * el servidor de sello de tiempo
 * @throws SignatureVerifyException  si no se ha podido realizar el proceso de verificación
 */
	public  boolean verify(InputStream contentStream, Signature signatureData)
          throws SignatureProviderException, IOException, SignatureVerifyException;

/**
 * Verifica la firma digital de un documento almacenado en un fichero de disco,  a partir de la firma encapsulada
 * en un objeto <code>Signature</code>
 * La verificación es independiente de si la firma llevaba o no sello de tiempo
 * @param fileName nombre del fichero del documento original
 * @param signatureData objeto que contiene la firma
 * @return <code>true</code> si la verificación es correcta y <code>false</code> en caso contrario
 * @throws FileNotFoundException  si no existe el fichero del documento a verificar
 * @throws SignatureProviderException  si no se ha podido acceder a la API del proveedor de firma electrónica
 * @throws IOException  si no se ha podido acceder al fichero o si no se ha podido contactar con
 * el servidor de sello de tiempo
 * @throws SignatureVerifyException si no se ha podido realizar el proceso de verificación
 */
  public  boolean verify(String fileName, Signature signatureData) throws 
          FileNotFoundException, SignatureProviderException, IOException, SignatureVerifyException;
  

/**
 * Verifica la firma digital de un documento de una URL a partir de la firma encapsulada
 * en un objeto <code>Signature</code>
 * La verificación es independiente de si la firma llevaba o no sello de tiempo
 * @param url dirección URL del fichero del documento original
 * @param signatureData objeto que contiene la firma
 * @return <code>true</code> si la verificación es correcta y <code>false</code> en caso contrario
 * @throws FileNotFoundException  si no existe el fichero del documento a verificar
 * @throws SignatureProviderException  si no se ha podido acceder a la API del proveedor de firma electrónica
 * @throws IOException  si no se ha podido acceder al fichero o si no se ha podido contactar con
 * el servidor de sello de tiempo
 * @throws SignatureVerifyException si no se ha podido realizar el proceso de verificación
 */
  public  boolean verify(URL url, Signature signatureData) throws 
          SignatureProviderException, IOException, SignatureVerifyException;
  
  /**
   * Obtiene el documento SMIME a partir del documento original y su firma digital
   * @throws java.io.IOException
   * @param smime OutputStream con el SMIME obtenido
   * @param signature firma digital del documento 
   * @param document InputStream con el documento que se firmó
   */
  
  public void generateSMIME (InputStream document, Signature signature, OutputStream smime) throws IOException;
  
  
  /**
   * Obtiene el documento SMIME a partir del documento original y el conjunto de firmas digitales obtenidas de firmas el documento.
   * @param document InputStream con el documento que se firmó 
   * @param signatures firmas digital del documento 
   * @param smime  OutputStream con el SMIME obtenido
   * @throws IOException
   */
   public void generateSMIMEParalell(InputStream document, Signature[] signatures, OutputStream smime) throws IOException, SignatureException;
  
  /** 
   * Obtiene la hora oficial que tendría un sello de tiempo generado en ese mismo instante
   * 
   * @return Hora Oficial
 * @param certificateName nombre del certificado que se usará para firmar
 * @param password contraseña de la clave privada del usuario
 * @param contentType tipo MIME del documento a firmar
 * @throws IOException
 * @throws SignatureException
   */
  public Date getCurrentDate(String certificateName, String password, String contentType) throws SignatureTimestampException, SignatureException, IOException;

  /**
   * Obtiene la versión del componente
   * @return versionNumber
   */
  public String getAPIVersion ();
  
  
  /**
   * Devuelve el un SMIMEParser con el que se puede obtener información de un documento smime
   * 
   * @param smime el documento smime a interpretar
   * @return El intérprete que nos permite obtener información del smime.
   * @throws FileNotFoundException
   * @throws InstantiationException
   * @throws IllegalAccessException
   * @throws IOException
   * @throws SignatureException
   */
  
  public SMIMEParser getSMIMEParser(InputStream smime) throws FileNotFoundException, InstantiationException, IllegalAccessException, IOException, SignatureException;
  
  /**
   * Devuelve un objeto con información acerca del certificado.
   * 
   * @param certificate Certificado que se quiere parsear.
   * @return Certificado parseado.
   */
  public ParsedCertificate parseCertificate(X509Certificate certificateChain);

}
