package es.caib.signatura.api;

import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.net.URL;
import java.util.Map;
import java.util.Properties;

import es.caib.signatura.impl.CAIBSigner;
import es.caib.signatura.impl.SigUtil;

/**
 * Clase factory que obtiene la clase que implementa la interfaz <code>Signature</code> en función
 * del nombre de la entidad certificadora
 * @author Jesús Reyes (3dígits)
 * @version 1.0
 * @see Signature
 */

public class SignerFactory
{

	static boolean	started	= false;
	private static Map signerConfiguration = null;
	
	public SignerFactory() {}
	
	public SignerFactory(Map signerConfig) {
		signerConfiguration = signerConfig;
	}
	
	public String getAPIVersion()
		throws FileNotFoundException, IOException
	{
		InputStream inputStream = SignerFactory.class.getResourceAsStream("version.properties");
		if (inputStream == null) {
			throw new FileNotFoundException();
		}
		Properties tempProperties = new Properties();
		tempProperties.load(inputStream);

		inputStream.close();
		return tempProperties.getProperty("Version");
	}
	
	public URL getUpdateSite() throws FileNotFoundException, IOException {
		InputStream inputStream = SignerFactory.class.getResourceAsStream("version.properties");
		if (inputStream == null) {
			throw new FileNotFoundException();
		}
		Properties tempProperties = new Properties();
		tempProperties.load(inputStream);
		inputStream.close();
		String updateSite = tempProperties.getProperty("updateSite");
		if (updateSite != null && !updateSite.equals(""))
			return new URL(updateSite);
		else {
			return null;
		}
	}

	/**
	 * Obtiene un objeto de la clase que implementa la interfaz <code>Signature</code> según
	 * el nombre de la entidad certificadora
	 * @return objeto que implementa <code>Signature</code> para la entidad certificadora <code>caName</code>
	 * @throws UpgradeNeededException 
	 */
	public Signer getSigner()
		throws UpgradeNeededException
	{
		try {
			if (!started) {
				started = true;
			}
			Signer s = null;
			if (signerConfiguration != null) {
				s = new CAIBSigner(signerConfiguration);
			}
			else {
				s = new CAIBSigner();
			}
			if (!isValidVersion(getAPIVersion(), s.getAPIVersion())) {
				System.out.println("Necesita actualizarse a la versión " + getAPIVersion()
					+ " o superior. Actualmente tiene instalada la versión " + s.getAPIVersion());
				throw new UpgradeNeededException("Necesita actualizarse a la versión " + getAPIVersion()
					+ " o superior. Actualmente tiene instalada la versión " + s.getAPIVersion(),
					SigUtil.getUpgradeUrl());
			}
			return s;
		} catch (Throwable e) {
			throw new UpgradeNeededException("Sistema de firma no instalado", SigUtil.getUpgradeUrl(), e);
		}
	}

	private boolean isValidVersion( String apiVersion, String jreVersion )
	{
		if (jreVersion == null) {
			return false;
		}
		if (apiVersion == null) {
			return false;
		}
		String splitJRE[] = jreVersion.split("[.-]");
		String splitAPI[] = apiVersion.split("[.-]");
		
		/**
		 * modifiquem les condicions per a saber si és versió vàlida.
		 * Ara no tenir sub-versió es considera com a versió -1.
		 */
		for (int i = 0; i < Math.max(splitAPI.length,splitJRE.length) ; i++) {
			int v1;
			int v2;
			try {
				v1 = (i<splitJRE.length)?Integer.parseInt(splitJRE[i]):-1; //si no s'ha definit aquest nivell de versió, el considerem -1
				v2 = (i<splitAPI.length)?Integer.parseInt(splitAPI[i]):-1; //si no s'ha definit aquest nivell de versió, el considerem -1
				if (v1 > v2) { // Versión superior
					return true;
				}
				if (v1 < v2) { // Versión inferior
					return false;
				}
			} catch (NumberFormatException e) {
				if (i >= splitJRE.length) {
					return false;
				}
						
				if (splitJRE[i].compareTo(splitAPI[i]) > 0)
					return true;
				if (splitJRE[i].compareTo(splitAPI[i]) < 0)
					return false;
			}
		}
		return true;
	}
}
