package es.caib.signatura.impl;

import java.io.IOException;
import java.io.InputStream;
import java.io.Serializable;
import java.security.cert.X509Certificate;
import java.util.Date;

import es.caib.signatura.api.SignatureVerifyException;

import es.caib.signatura.api.ParsedCertificate;
import es.caib.signatura.api.Signature;
import es.caib.signatura.api.SignatureDataException;
import es.caib.signatura.api.SignatureProviderException;
import es.caib.signatura.api.SignatureTimestampException;
import es.caib.signatura.api.Signer;
import es.caib.signatura.impl.ClassLoaderFactory;
import es.caib.signatura.impl.SignatureProviderInterface;

/**
 * Implementación de la interfaz <code>SignatureData</code> para usar con la
 * entidad certificadora Tradisea
 * 
 * @author Jesús Reyes (3dígits)
 * @version 0.98
 * @see Signer
 * @see Signature
 * 
 */
public class CMSSignature implements Signature, Serializable {

	protected byte[] signatureBytes = null; // array de bytes con la firma
		
	private String contentType = null;

	private static final long serialVersionUID = 1;

	private transient SignatureProviderInterface impl = null;

	/**
	 * Crea un nuevo objeto a partir de los atributos de la clase. Es el
	 * constructor que debe usar cada implementación de la interfaz
	 * <code>Signature</code> para crear una firma. Se extrae el certificado
	 * de la firma y se guarda en la propiedad <code>transient</code>
	 * certificate para usarla en los métodos que dan información concreta del
	 * certificado
	 * 
	 * @param signatureBytes
	 *            array de bytes con la firma digital generada por la api del
	 *            proveedor de firma electrónica
	 * 
	 */
	public CMSSignature(byte pkcs7[], String contentType)
			throws SignatureDataException {
		signatureBytes = pkcs7;
		this.contentType = contentType;
		try {
			init();
		} catch (IOException e) {
			throw new SignatureDataException(e);
		}
	}
	
	public void setContentType(String contentType){		
		this.contentType = contentType;
	}
	
	public void setPkcs7(byte pkcs7[]){		
		signatureBytes = pkcs7;
	}
	
	public CMSSignature (SignatureProviderInterface impl)
	{
		this.impl = impl;
		signatureBytes = impl.getPkcs7();
		contentType = impl.getContentType();
	}

	private void init() throws IOException {
		if (impl == null) {
			try {
				ClassLoader cl = ClassLoaderFactory.getFactory().getMasterClassLoader();
				Class clazz = cl .loadClass( getInternalClassName() );
				impl =(SignatureProviderInterface)clazz.newInstance();
			} catch (InstantiationException e) {
				throw new RuntimeException(e);
			} catch (IllegalAccessException e) {
				throw new RuntimeException(e);
			} catch (ClassNotFoundException e) {
				throw new RuntimeException(e);
			}
		}
		try {
			impl.setContentType(contentType);
			impl.setSignedData(signatureBytes);
		} catch (Exception e) {
			throw new IOException("Unable to parse signature");
		}
	}

	/**
	 * @return
	 */
	protected String getInternalClassName() {
		return "es.caib.signatura.provider.impl.common.CMSSignatureImpl";
	}

	/**
	 * Obtiene el nombre de la entidad certificadora usada en la firma
	 * 
	 * @return nombre de la entidad certificadora
	 */
	public String getCertCaName() {
		return impl.getCertCaName();
	}

	/**
	 * Obtiene el nombre del certificado usado en la firma
	 * 
	 * @return nombre del certificado (CommonName)
	 */
	public String getCertSubjectCommonName() {
		return impl.getCertSubjectCommonName();
	}

	public String getCertSubjectAlternativeNames() {
		return impl.getCertSubjectAlternativeNames();
	}

	public byte[] getPkcs7() {
		return this.signatureBytes;
	}

	// Sobreescribimos el método readObject que deserializa el objeto para
	// llamar posteriormente al método privado de inicialización del objetco
	// cuyo cometido es extraer el certificado de usuario de la firma
	private void readObject(java.io.ObjectInputStream in) throws IOException,
			ClassNotFoundException {
		in.defaultReadObject();
		init();
	}

	public Date getDate() throws SignatureTimestampException {
		return impl.getDate();
	}

	public boolean verify() throws SignatureVerifyException {
		return impl.verify();
	}

	public String getContentType() {
		return contentType;
	}

	public X509Certificate getCert() {
		return impl.getCert();
	}

	public X509Certificate[] getCertificateChain() throws Exception {
		return impl.getCertificateChain();
	}

	public ParsedCertificate getParsedCertificate() {
		return impl.getParsedCertificate();
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see es.caib.signatura.api.Signature#verify(java.io.InputStream)
	 */

	public boolean verify(InputStream contentStream)
			throws SignatureProviderException, IOException,
			SignatureVerifyException {
		boolean isVerified = true;
		try{
			// verificación interna de la firma
			isVerified = isVerified && impl.verify(contentStream);
			// isVerified = isVerified && verifyWS();
		}catch(SignatureVerifyException e){
			throw e;
		} catch(Exception e){
			throw new SignatureVerifyException(e);
		}
		return isVerified;
	}


	public boolean verifyAPosterioriTimestamp(InputStream contentStream) throws SignatureProviderException, IOException,
			SignatureVerifyException {
		boolean isVerified = true;
		try{			
			// verificación interna de la firma
			isVerified = isVerified && impl.verifyAPosterioriTimestamp(contentStream);
			//*
			// * Se actualiza la firma por si el validador ha añadido
			// * un sello de tiempo

			if(isVerified){
				this.setPkcs7(impl.getPkcs7());
			}

		}catch(SignatureVerifyException e){
			throw e;
		} catch(Exception e){
			throw new SignatureVerifyException(e);
		}
		return isVerified;
	}
	/*
	private boolean WSResponseIsAVerifiedMessage(String result) throws SignatureVerifyException {
		String content = new String();
		try {
			DocumentBuilderFactory documentBuilderFactory = DocumentBuilderFactory.newInstance();
			DocumentBuilder documentBuilder = documentBuilderFactory.newDocumentBuilder();
			Document document;	    	  
			StringBufferInputStream inputStream = new StringBufferInputStream(result);
			document = documentBuilder.parse((InputStream)inputStream);	    	  
			//Element element = document.getDocumentElement();
			try{
				NodeList nodeList = document.getElementsByTagName("valido");
				Node node = nodeList.item(0);
				Element element = (Element) node;
				String tagName = element.getTagName();
				Node textData = null;
				boolean stop = false;
				NodeList nodeDataList = element.getChildNodes();				
				for(int i = 0;i < nodeDataList.getLength() && !stop;i++){
					textData = nodeDataList.item(i);
					stop = textData.getNodeType() == org.w3c.dom.Node.TEXT_NODE;						
				}
				content = ((org.w3c.dom.Text)textData).getData();
			}catch(Exception e){
				//* Para no llevar a malos entendidos
				// * se da la firma como falsa
				// *
				return false;
			}
			if(content == null){
				throw new Exception("No content in 'valido' TAG. Response message: " + result);
			}
		}catch(Exception e){
			throw new SignatureVerifyException(e);
		}
		return content.compareToIgnoreCase("true") == 0;
	}
	 */	
}
