package es.caib.signatura.impl;

import java.io.*;
import java.lang.reflect.Constructor;
import java.security.InvalidKeyException;
import java.security.NoSuchAlgorithmException;
import java.security.NoSuchProviderException;
import java.security.SignatureException;
import java.security.cert.CertificateException;
import java.security.cert.CertificateExpiredException;
import java.security.cert.CertificateNotYetValidException;
import java.security.cert.X509Certificate;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;

import es.caib.signatura.api.CertificateVerifyException;
import es.caib.signatura.api.ParsedCertificate;
import es.caib.signatura.api.SignatureProviderException;
import es.caib.signatura.api.SignatureVerifyException;
import es.caib.signatura.api.Certificate;

public class CertificateImpl implements Certificate {
	private X509Certificate[] certificateChain = null;
	
	/*
	 * Constructora de la clase
	 */
	public CertificateImpl(X509Certificate[] certificateChain){
		this.certificateChain = certificateChain;
	}
	
	/**
	 * Obtiene el nombre de la entidad certificadora
	 * @return nombre de la entidad certificadora
	 */
	  public String getCertCaName(){
		if(certificateChain == null || certificateChain.length == 0){
			throw new Error("No se encuentra cadena de certificación.");
		}
		return certificateChain[certificateChain.length-1].getSubjectX500Principal().getName();
	  }

	/**
	 * Obtiene el nombre del certificado
	 * @return nombre del certificado (CommonName)
	 */

	  public String getCertSubjectCommonName(){
		  ParsedCertificate Parsed = this.getParsedCertificate();
		  return(Parsed.getName());
	  }

	/**
	 * Obtiene la concatenación del SubjectAlternateName del certificado del usuario en la forma
	 * nombre0 = valor, nombre1 = valor, ...
	 * @return cadena con el SubjectAlternateName del certificado del usuario
	 */

	  public String getCertSubjectAlternativeNames(){
		  StringBuffer altNameSB = new StringBuffer("");
			String altNameString = null;
			try {
				Collection generalNames = certificateChain[0].getSubjectAlternativeNames();
				Iterator itr = generalNames.iterator();
				while (itr.hasNext()) {
					List list = (List) itr.next();

					int tagNo = ((Integer) list.get(0)).intValue();
					switch (tagNo) {
					case 0:
						altNameSB.append("," + "otherName=");
						break;

					case 1:
						altNameSB.append("," + "rfc822Name=");
						break;

					case 2:
						altNameSB.append("," + "dNSName=");
						break;

					case 3:
						altNameSB.append("," + "x400Address=");
						break;

					case 4:
						altNameSB.append("," + "directoryName=");
						break;

					case 5:
						altNameSB.append("," + "ediPartyName=");
						break;

					case 6:
						altNameSB.append("," + "uniformResourceIdentifier=");
						break;

					case 7:
						altNameSB.append("," + "iPAddress=");
						break;

					case 8:
						altNameSB.append("," + "registeredID=");
						break;

					}
					altNameSB.append(list.get(1).toString());

					// Quitamos la coma del principio
					if (altNameSB.length() > 0) {
						altNameString = altNameSB.substring(1, altNameSB.length());
					}

				}
			}

			catch (Exception ex) {
				return null;
			}

			return altNameString;
	  }


	  /**
	   * Devuelve el certificado X509 
	   * @return X509Certificate
	   */  
	  public X509Certificate getCert(){
		  return certificateChain[0];
	  }
	  
	  /**
	   * Devuelve el Seycon Principal a partir del certificado
	   * @return SeyconPrincipal
	   */  
	  public ParsedCertificate getParsedCertificate(){
		try {
			ClassLoader cl = ClassLoaderFactory.getFactory().getMasterClassLoader();
			Class clazz = cl .loadClass( "es.caib.signatura.provider.impl.common.ParsedCertificatImpl" );
			Constructor constructor = clazz.getConstructor(new Class[] {certificateChain.getClass(), Boolean.TYPE});
			ParsedCertificate parsed = (ParsedCertificate) constructor.newInstance(new Object [] {certificateChain, new Boolean(false)});
			return new ParsedCertificateProxy (parsed);
		} catch (Exception e) {
			throw new RuntimeException(e);
		}
	  }
	  
	  /**
	   * 
	   * Verifica el certificado y su cadena de certificación 
	   * @param contentStream flujo de bytes del certificado original
	   * @return <code>true</code> si la verificación es correcta y <code>false</code> en caso contrario
	   * @throws SignatureProviderException  si no se ha podido acceder a la API del proveedor de firma electrónica
	   * @throws IOException  si no se ha podido acceder al fichero o si no se ha podido contactar con
	   * el servidor de sello de tiempo
	   * @throws SignatureVerifyException  si no se ha podido realizar el proceso de verificación
	   */
	  	public boolean verify()
	            throws IOException, CertificateVerifyException{
	  		boolean isValid = true;
	  		isValid = isValid && verifyCertificateChain();
	  		if(!isValid){
	  			/*
	  			 * Podría darse el caso que la cadena de certificación estuviera invertida
	  			 */
	  			invertCertificateChain();
	  			isValid = isValid && verifyCertificateChain();
	  			if(!isValid){
	  				/*
	  				 * Si no es válida la cadena de certificación una vez invertida se asume
	  				 * que la cadena de certificación original es la válida
	  				 */
	  				invertCertificateChain();
	  			}
	  		}
	  		return isValid;
	  	}
	  	
	  	/**
	  	 * Operación que invierte la cadena de certificación que contiene la clase 
	  	 */
		private void invertCertificateChain(){
			for(int i = 0;i < certificateChain.length / 2;i++){
				X509Certificate temporalCertificate = certificateChain[certificateChain.length - 1 - i];
				certificateChain[certificateChain.length - 1 - i] = certificateChain[i];
				certificateChain[i] = temporalCertificate;
			}
		}
		
	  /**
	   * 
	   * Verifica el certificado y su cadena de certificación 
	   * @param contentStream flujo de bytes del certificado original
	   * @return <code>true</code> si la verificación es correcta y <code>false</code> en caso contrario
	   * @throws SignatureProviderException  si no se ha podido acceder a la API del proveedor de firma electrónica
	   * @throws IOException  si no se ha podido acceder al fichero o si no se ha podido contactar con
	   * el servidor de sello de tiempo
	   * @throws SignatureVerifyException  si no se ha podido realizar el proceso de verificación
	   */
	  	private boolean verifyCertificateChain()
	            throws IOException, CertificateVerifyException{
	  		boolean isValid = true;
	  		/*
	  		 * verificación basada en fechas
	  		 */
	  		for(int i = 0;i < certificateChain.length && isValid;i++){
	  			try {
	  				certificateChain[i].checkValidity();
	  			} catch (CertificateExpiredException cee) {
	  				isValid = false;
	  			} catch (CertificateNotYetValidException cve) {
	  				isValid = false;
	  			}
	  		}	  		
	  		/*
	  		 * verificación de firmas
	  		 */
	  		for(int i = 0;(i < certificateChain.length - 1) && isValid;i++){	  			
	  				try {
						certificateChain[i].verify(certificateChain[i + 1].getPublicKey());
					} catch (InvalidKeyException e) {
						isValid = false;
					} catch (CertificateException e) {
						isValid = false;
					} catch (NoSuchAlgorithmException e) {
						throw new CertificateVerifyException(e);
					} catch (NoSuchProviderException e) {
						throw new CertificateVerifyException(e);
					} catch (SignatureException e) {
						isValid = false;
					}	  			
	  		}
	  		/*
	  		 * verificación mediante Web Services
	  		 * En dicha verificación se valida el certificado raiz
	  		 */
	  		try{
	  			isValid = isValid && verifyCertificateWebServices(certificateChain);
	  		}catch(Exception e){
	  			throw new CertificateVerifyException(e);
	  		}
	  		return isValid;
	  	}
	  	
	  	/*
	  	 * Validación mediante Web Services de la cadena de certificación
	  	 */
	  	private boolean verifyCertificateWebServices(X509Certificate[] certificateChain)throws CertificateVerifyException{
	  		/* TODO: commons-logging*/
	  		return true;
	  		/*
	  		String result = verifyWebServices(certificateChain);
  			return WSResponseIsAVerifiedMessage(result);
  			*/
	  	}
	  	
		/*
		 * Validación mediante Web Services de la cadena de certificación
		 */
	  	/*
		private String verifyWebServices(X509Certificate[] certificateChain) throws CertificateVerifyException{
			String toReturn = new String();
			try{
				URLClassLoader urlClassLoader = (URLClassLoader)ClassLoaderFactory.getFactory().getMasterClassLoader();
				Class c;			
				c = urlClassLoader.loadClass("es.caib.signatura.cliente.ValidadorCertificadosEmbedded");						
				Constructor constructor = c.getConstructor(new Class [] {});
				IValidadorCertificados iValidadorCertificados = (IValidadorCertificados)constructor.newInstance(new Object [] {});
				//ValidadorCertificadosEmbedded validadorCertificados = new ValidadorCertificadosEmbedded();
				toReturn = iValidadorCertificados.validarCertificadoParaFirma(certificateChain);
			} catch (Exception e) {
				throw new CertificateVerifyException(e);
			}
			return toReturn;
		}
	  	
		private boolean WSResponseIsAVerifiedMessage(String result) throws CertificateVerifyException {
			String content = new String();
			try {
				DocumentBuilderFactory documentBuilderFactory = DocumentBuilderFactory.newInstance();
				DocumentBuilder documentBuilder = documentBuilderFactory.newDocumentBuilder();
				Document document;	    	  
				StringBufferInputStream inputStream = new StringBufferInputStream(result);
				document = documentBuilder.parse((InputStream)inputStream);	    	  
				//Element element = document.getDocumentElement();
				try{
					NodeList nodeList = document.getElementsByTagName("validado");
					Node node = nodeList.item(0);
					Element element = (Element) node;
					String tagName = element.getTagName();
					Node textData = null;
					boolean stop = false;
					NodeList nodeDataList = element.getChildNodes();				
					for(int i = 0;i < nodeDataList.getLength() && !stop;i++){
						textData = nodeDataList.item(i);
						stop = textData.getNodeType() == org.w3c.dom.Node.TEXT_NODE;						
					}
					content = ((org.w3c.dom.Text)textData).getData();
				}catch(Exception e){
					// Para no llevar a malos entendidos
					// se da la firma como falsa
					// 
					return false;
				}
				if(content == null){
					throw new Exception("No content in 'validado' TAG. Response message: " + result);
				}
			}catch(Exception e){
				throw new CertificateVerifyException(e);
			}
			return content.compareToIgnoreCase("true") == 0;
		}
	  	*/

}
