
package es.caib.signatura.impl;
import java.io.IOException;
import java.io.InputStream;
import java.io.UnsupportedEncodingException;

/** 
 * Clase que transforma una firma pkcs7 en un InputStream en formato SMIME.
 * En concreto, el formato SMIME devuelto es application/pkcs7-mime.
 *  
 * @author  3digits
 * @version 1.0
 */
public class SMIMEPkcs7InputStream extends InputStream {

	private static final int CHUNK_SIZE = 48;
  
  
  
    /** Flag que indica si se han leido todos los bytes de la entrada */
	private int status;
	static private final int STATUS_IDLE = 0;
	static private final int STATUS_HEADER = 1;
	static private final int STATUS_BODY = 2;
	static private final int STATUS_END = 3;
	
	int bufferOffset;
	byte buffer [];

	byte signatureData [];
	int signatureDataOffset ;
	
  /**
   * Crea un nuevo SMIMEInputStream a partir de la firma en pkcs7 y el contentType del MIME
   * @param pkcs7 el array de bytes que representa el pkcs7
   */
    public SMIMEPkcs7InputStream(byte[] pkcs7) {
      status = STATUS_IDLE;
      bufferOffset = 0;
      buffer = null;
      signatureData = pkcs7;
      signatureDataOffset = 0;
      
    }
      
   
    
    /* (non-Javadoc)
     * @see java.io.InputStream#close()
     */
    public synchronized void close ()
    throws IOException {
        status = STATUS_END;
    }

    /* (non-Javadoc)
     * @see java.io.InputStream#read()
     */
    public synchronized int read ()
    throws IOException {
    	do {
    		if ( buffer != null && bufferOffset < buffer.length)
	    	{
	    		byte b = buffer [ bufferOffset ++ ];
	    		return b;
	    	} else if (status == STATUS_END) {
    			return -1;
	    	} else {
	    		fetchData();
	    	}
	    	
    	} while (true);
    }

    /* (non-Javadoc)
     * @see java.io.InputStream#read(byte[], int, int)
     */
    public synchronized int read (byte[] targetBuffer, int offset, int length)
    throws IOException {
    	do {
    		if ( buffer != null && bufferOffset < buffer.length)
	    	{
    			int actualLength = buffer.length - bufferOffset;
    			if ( actualLength > length)
    				actualLength = length;
    			System.arraycopy(buffer, bufferOffset, targetBuffer, offset, actualLength);
    			bufferOffset = bufferOffset + actualLength;
    			return actualLength;
	    	} else if (status == STATUS_END) {
        		return -1;
	    	} else {
	    		fetchData();
	    	}
	    	
    	} while (true);
    }

    /* (non-Javadoc)
     * @see java.io.InputStream#available()
     */
    public synchronized int available()
    throws IOException {
    	do {
    		 if ( buffer != null && bufferOffset < buffer.length)
	    	{
    			return buffer.length - bufferOffset;
	    	} else if (status == STATUS_END) {
	    		return -1;
	    	} else {
	    		fetchData ();
	    	}
    	} while (true);
    }

	private void fetchData() throws IOException {
		switch (status)
		{
		case STATUS_IDLE:
			buffer = fetchIdle ();
			break;
		case STATUS_HEADER:
			buffer = fetchHeader ();
			break;
		case STATUS_BODY:
			buffer = fetchSignatureBody ();
			break;
		default:
			buffer = null;
			break;
		}
		bufferOffset = 0;
	}

	private byte [] fetchSignatureBody() {
		int len = signatureData.length - signatureDataOffset;
		if ( len <= 0)
		{
			status = STATUS_END;
			return null;
		}
		else if ( len > CHUNK_SIZE )
		{
			len = CHUNK_SIZE;
		}
		
		byte result []= Base64.toBase64(signatureData, signatureDataOffset, len);
		signatureDataOffset = signatureDataOffset + len;
		return result;
	}


	private byte [] fetchHeader() {
    	try {
        	status = STATUS_BODY;
			return ("Content-Type: application/pkcs7-mime; smime-type=signed-data; name=smime.p7m;\r\n"+
					"Content-Transfer-Encoding: base64\r\n"+
					"Content-Disposition: attachment; filename=\"smime.p7m\"\r\n"+
					"Content-Description: S/MIME Cryptographic Signature\r\n"+
					"\r\n").getBytes("ISO-8859-1");
		} catch (UnsupportedEncodingException e) {
			throw new RuntimeException (e);
		}
	}


	private byte [] fetchIdle() {
		status = STATUS_HEADER;
		return null;
	}


	

  
    /* (non-Javadoc)
     * @see java.io.InputStream#skip(long)
     */
    public synchronized long skip(long n) throws IOException {
        for (long i = 0; i < n; i++)
            if (this.read() < 0) return i;
        return n;
    }

    /* (non-Javadoc)
     * @see java.io.InputStream#mark(int)
     */
    public void mark(int readlimit) {}

    /* (non-Javadoc)
     * @see java.io.InputStream#reset()
     */
    public void reset() throws IOException {
        throw new IOException("Base64InputStream does not support mark/reset");
    }

    /* (non-Javadoc)
     * @see java.io.InputStream#markSupported()
     */
    public boolean markSupported() { return false; }


// Own methods
    
    
}
