package es.caib.signatura.impl;

import java.io.IOException;
import java.io.InputStream;
import java.security.cert.X509Certificate;
import java.util.Date;

import es.caib.signatura.api.ParsedCertificate;
import es.caib.signatura.api.Signature;
import es.caib.signatura.api.SignatureDataException;
import es.caib.signatura.api.SignatureProviderException;
import es.caib.signatura.api.SignatureTimestampException;
import es.caib.signatura.api.SignatureVerifyException;
import es.caib.signatura.impl.ClassLoaderFactory;
import es.caib.signatura.impl.SignatureProviderInterface;

/**
 * Proxy de acceso a la implementación de la firma SMIME.
 * @author u91940
 *
 */

public class SMIMESignatureProxy implements Signature {

	protected byte[] signatureBytes = null; // array de bytes con la firma
		
	private String contentType = null;

	private static final long serialVersionUID = 1;

	private transient SignatureProviderInterface impl = null;

	/**
	 * Crea un nuevo objeto a partir de los atributos de la clase. Es el
	 * constructor que debe usar cada implementación de la interfaz
	 * <code>Signature</code> para crear una firma. Se extrae el certificado
	 * de la firma y se guarda en la propiedad <code>transient</code>
	 * certificate para usarla en los métodos que dan información concreta del
	 * certificado
	 * 
	 * @param signatureBytes
	 *            array de bytes con la firma digital generada por la api del
	 *            proveedor de firma electrónica
	 * 
	 */
	public SMIMESignatureProxy(byte pkcs7[], String contentType)
			throws SignatureDataException {
		signatureBytes = pkcs7;
		this.contentType = contentType;
		try {
			init();
		} catch (IOException e) {
			throw new SignatureDataException(e);
		}
	}
	
	

	private void init() throws IOException {
		if (impl == null) {
			try {
				ClassLoader cl = ClassLoaderFactory.getFactory().getMasterClassLoader();
				Class clazz = cl .loadClass( getInternalClassName() );
				impl =(SignatureProviderInterface)clazz.newInstance();
			} catch (InstantiationException e) {
				throw new RuntimeException(e);
			} catch (IllegalAccessException e) {
				throw new RuntimeException(e);
			} catch (ClassNotFoundException e) {
				throw new RuntimeException(e);
			}
		}
		try {
			impl.setContentType(contentType);
			impl.setSignedData(signatureBytes);
		} catch (Exception e) {
			throw new IOException("Unable to parse signature");
		}
	}

	/**
	 * @return
	 */
	protected String getInternalClassName() {
		return "es.caib.signatura.provider.impl.common.SMIMESignatureImpl";
	}

	/**
	 * Obtiene el nombre de la entidad certificadora usada en la firma
	 * 
	 * @return nombre de la entidad certificadora
	 */
	public String getCertCaName() {
		return impl.getCertCaName();
	}

	/**
	 * Obtiene el nombre del certificado usado en la firma
	 * 
	 * @return nombre del certificado (CommonName)
	 */
	public String getCertSubjectCommonName() {
		return impl.getCertSubjectCommonName();
	}

	public String getCertSubjectAlternativeNames() {
		return impl.getCertSubjectAlternativeNames();
	}


	public Date getDate() throws SignatureTimestampException {
		return impl.getDate();
	}

	public boolean verify() throws SignatureVerifyException {
		return impl.verify();
	}

	public String getContentType() {
		return contentType;
	}

	public X509Certificate getCert() {
		return impl.getCert();
	}

	public X509Certificate[] getCertificateChain() throws Exception {
		return impl.getCertificateChain();
	}

	public ParsedCertificate getParsedCertificate() {
		return impl.getParsedCertificate();
	}



	public boolean verify(InputStream contentStream)
			throws SignatureProviderException, IOException,
			SignatureVerifyException {
		boolean isVerified = true;
		try{
			// verificación interna de la firma
			//TODO comprobar si ha de ser RAW o no per cridar a verifyRaw o a verify
			isVerified = isVerified && impl.verify(contentStream);
		
		}catch(SignatureVerifyException e){
			throw e;
		} catch(Exception e){
			throw new SignatureVerifyException(e);
		}
		return isVerified;
	}



	public byte[] getPkcs7() {
		return null;
	}



	public boolean verifyAPosterioriTimestamp(InputStream contentStream) throws SignatureProviderException, IOException, SignatureVerifyException {
		boolean isVerified = true;
		try{
			// verificación interna de la firma
			//TODO comprobar si ha de ser RAW o no per cridar a verifyRaw o a verify
			isVerified = isVerified && impl.verifyAPosterioriTimestamp(contentStream);
		
		}catch(SignatureVerifyException e){
			throw e;
		} catch(Exception e){
			throw new SignatureVerifyException(e);
		}
		return isVerified;	
	}



}
