
package es.caib.signatura.impl;


import java.io.InputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.OutputStream;
import java.security.UnrecoverableKeyException;
import java.util.Date;

import es.caib.signatura.api.SignaturePrivKeyException;

import es.caib.signatura.api.Signature;
import es.caib.signatura.api.SignatureCertNotFoundException;
import es.caib.signatura.api.SignatureException;
import es.caib.signatura.api.SignatureProviderException;
import es.caib.signatura.api.SignatureSignException;
import es.caib.signatura.api.SignatureTimestampException;

/**Interfaz que independiza las implementaciones de firma y verificación de cada entidad certificadora con una
 * serie de métodos estándar, llevando a cabo la firma digital desacoplada (firma y documento original
 * se mantienen por separado). Esta firma puede ser avanzada o avanzada con sello de tiempo.
 * La firma se encapsula con la clase <code>Signature</code>.
 * 
 * @author 3dígits
 * @version 1.0
 * @see Signature
 */
public interface SignerProviderInterface {

	

/**
 * Obtiene la lista de certificados disponibles en el almacén de certificados definido por la API 
 * de la entidad certificadora: disco duro, dispositivo USB, etc.
 * @return lista de los nombres de certificado disponibles
 * @throws SignatureCertNotFoundException  si no se encuentra ningún certificado disponible
 * @throws SignaturePrivKeyException  si no se pueden obtener las claves privadas de los certificados
 */
	public  String[] getCertList(boolean recognized) throws SignatureCertNotFoundException, SignaturePrivKeyException;

  /**
   * Firma digitalmente de un documento PDF pasado como <code>InputStream</code> y devuelve el mismo PDF firmado
   * y modificado de forma que, en uno de los bordes del documento que se le indique, aparezca el firmante, 
   * la url desde la que se puede consultar el PDF y una matriz de puntos en formato PDF417 que continene 
   * esa misma URL. 
   * @param contentStream flujo de bytes del documento PDF a firmar
   * @param certificateName nombre del certificado que se usará para firmar
   * @param password contraseña de la clave privada del usuario
   * @param contentType tipo MIME del documento a firmar
   * @param url URL que se sobreimpresiona sobre el documento firmado
   * @param position Posición en la que se se sobreimpresionará la url en cada una de las hojas del documento firmado. Pueden ser PDF_SIGN_POSITION_TOP,
   * PDF_SIGN_POSITION_BOTTOM, PDF_SIGN_POSITION_RIGHT o PDF_SIGN_POSITION_LEFT 
   * @return documento PDF firmado
   * @throws IOException  si ha habido algún problema de comunicaciÃ³n al pasar el stream de bytes
   * del documento
   * @throws SignatureProviderException  si no se ha podido acceder a la API del proveedor de firma electrónica
   * @throws SignatureSignException  si ha habido algún problema en el proceso de firma
   * @throws UnrecoverableKeyException  si la contraseña de la clave privada no es correcta
   * @throws SignatureException
   */
	public void signPDF (InputStream contentStream, OutputStream signedStream, String certificateName, String password, String contentType, 
			boolean recognized, String url, int position) 
    throws IOException, SignatureException;

/**
 * Obtiene la firma digital de un documento de disco y la encapsula en un objeto
 * <code>Signature</code> sin sello de tiempo.
 * @param fileName nombre del fichero que se desea firmar
 * @param certificateName nombre del certificado que se usará para firmar
 * @param password contraseña de la clave privada del usuario
 * @param contentType tipo MIME del documento a firmar
 * @return {@link Signature} firma del documento
 * @throws FileNotFoundException  si no se encuentra el fichero a firmar
 * @throws IOException  si ha habido algún problema al abrir el fichero
 * @throws SignatureProviderException  si no se ha podido acceder a la API del proveedor de firma electrónica
 * @throws SignatureSignException  si ha habido algún problema en el proceso de firma
 * @throws UnrecoverableKeyException  si la contraseña de la clave privada no es correcta
 * @throws SignatureException
 */
	public  Signature sign (InputStream contentStream, String certificateName, String password, String contentType, boolean recognized, boolean timeStamp, boolean rawSign) 
          throws IOException, SignatureException;


  /** 
   * Obtiene la hora oficial que tendría un sello de tiempo generado en ese mismo instante
   * 
   * @return Hora Oficial
 * @param certificateName nombre del certificado que se usará para firmar
 * @param password contraseña de la clave privada del usuario
 * @param contentType tipo MIME del documento a firmar
 * @throws IOException
 * @throws SignatureException
   */
  public Date getCurrentDate(String certificateName, String password, boolean recognized) throws SignatureTimestampException, SignatureException, IOException;

  /**
   * Obtiene la versión del componente
   * @return versionNumber
   */
  public String getVersion ();
}
