package es.caib.signatura.test.core;

import java.io.BufferedInputStream;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.ObjectOutputStream;
import java.io.OutputStream;
import java.io.PrintStream;
import java.io.PrintWriter;
import java.lang.reflect.Constructor;
import java.lang.reflect.InvocationTargetException;
import java.net.HttpURLConnection;
import java.net.URL;
import java.net.URLEncoder;
import java.util.ArrayList;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Vector;

import javax.swing.ImageIcon;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.xpath.XPath;
import javax.xml.xpath.XPathConstants;
import javax.xml.xpath.XPathExpressionException;
import javax.xml.xpath.XPathFactory;

import org.w3c.dom.Document;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.xml.sax.SAXException;

import es.caib.signatura.test.privileged.utils.SignatureTestUtils;

public class ExecutionDirector {

	public static final String ACTION_START_EXECUTIONS = "start-executions";
	public static final String ACTION_SHOW_RESULTS = "show-results";
	public static final String ACTION_EXPORT_RESULTS = "export-results";
	public static final String ACTION_DESTROY = "destroy";
	
	private Vector executions = new Vector();
	private ExecutionView view=null;
	private String config=null;
	private String configURL=null;
	private Map executionMap=new Hashtable();
	private SynchronizableBoolean loopQueryActions=new SynchronizableBoolean(true);
	private Map globalVariables=new Hashtable(); 
	private String monitorServletURL="";
	
	
	public void start(){
		setUp();
		while(loopQueryActions.get()){
			String action=view.queryActions();
			changeExecutionsStatus(Execution.STATUS_CONFIGURED);
			performInternalAction(action);
		}
	}
	
	private void changeExecutionsStatus(int statusConfigured) {
		Iterator it=executions.iterator();
		while(it.hasNext()){			
			Execution execution=(Execution)it.next();
			execution.setStatus(statusConfigured, getGlobalVariables());
		}
		
	}

	protected void performInternalAction(String action) {
		if(ACTION_START_EXECUTIONS.equals(action)){
			runExecutions();
			view.onFinalizeExecutions();
		}else if(ACTION_SHOW_RESULTS.equals(action)){
			view.updateResults(executions);
		}else if(ACTION_DESTROY.equals(action)){
			loopQueryActions.set(false);
			view.closeView();
		}else if(ACTION_EXPORT_RESULTS.equals(action)){
			String export=exportResults();
			view.exportResults(export);
		}
		
	}


	protected String exportResults() {
		StringBuffer export=new StringBuffer();
		
		//capçaleres
		export.append("Arq|SO|JRE|Nom|Descripció|Class|Resultat|Verificació\n");
		Iterator it=executions.iterator();
		int i=0;
		while(it.hasNext()){
			Execution ex=(Execution)it.next();
			
			if(ex.isActive()){
				export.append(System.getProperty("os.arch"));
				export.append('|');
				export.append(System.getProperty("os.name"));
				export.append('|');
				export.append(System.getProperty("java.version"));
				export.append('|');
				export.append(ex.getName());
				export.append('|');
				export.append(ex.getDescription());
				export.append('|');
				export.append(ex.getClass());
				export.append('|');
				
				//si no se ha validado anteriormente, validamos
				if(ex.getVerificationResult()==null){
					int result=ex.verify(getGlobalVariables());
					ex.setVerificationResult(new Integer(result));
				}
				if(ex.getVerificationResult().intValue()!=Execution.VERIFICATION_REMOTE){
					//verificación ok, ko o manual.
					if(ex.getVerificationResult().intValue()==Execution.VERIFICATION_MANUAL){
						try{
								String downloadURL=uploadExecutionResultFile(ex.getResult(), ex.getName(),ex.getResultContentType());
								export.append(downloadURL);
						}catch(Exception e){
							view.notifyFatal(e.getMessage(), e);
						}
					}else{
						if(ex.getResult()!=null)
							export.append(ex.getResult().toString());
						
					}
				}else{
					//verificación remota
					try{
						ex.setVerificationResult(new Integer(executeRemoteVerification(ex)));
					}catch(Exception e){
						view.notifyFatal(e.getMessage(), e);
					}
				
				}
				export.append('|');
				export.append(ex.getVerificationResult());
				export.append('\n');
				i++;
			}
		}
		return export.toString();
	}


	protected void runExecutions(){
		Iterator it=executions.iterator();
		while(it.hasNext()){			
			Execution execution=(Execution)it.next();
			
			if(execution.isActive()){
				try{
					execution.setStatus(Execution.STATUS_RUNNING,getGlobalVariables());
					PrintStream oldOut=System.out;
					PrintStream oldErr=System.err;
					SignatureTestUtils.setOut(execution.getOut());
					SignatureTestUtils.setErr(execution.getErr());
					execution.execute(getGlobalVariables());
					SignatureTestUtils.setOut(oldOut);
					SignatureTestUtils.setErr(oldErr);
					System.out.println(new String(execution.getOutStreamBytes().toByteArray()));
					System.err.println(new String(execution.getErrStreamBytes().toByteArray()));
					execution.setStatus(Execution.STATUS_DONE,getGlobalVariables());
				}catch(Exception e){
					e.printStackTrace();
					view.notifyFatal(e.getMessage(),e);
				}
			}
		}
	}


	
	protected void setUp(){
		if(view == null){
			view.notifyFatal("No view configured",new Exception("No view configured"));
		}
		
		try{
			parseXML(config);
			view.showAvailableExecutions(executions);
			changeExecutionsStatus(Execution.STATUS_CREATED);
			executions=view.queryUserExecutionsConfig();
		}catch(Exception e){
			e.printStackTrace();
			view.notifyFatal(e.getMessage(),e);
		}
	}
	
	protected void parseXML(String xml) throws Exception{
        DocumentBuilder builder = DocumentBuilderFactory.newInstance().newDocumentBuilder();
        ByteArrayInputStream bis=new ByteArrayInputStream(xml.getBytes());
        Document doc=builder.parse(bis);
        bis.close();
        
        XPathFactory factory = XPathFactory.newInstance();
        XPath xpath = factory.newXPath();
        
        
        parseGlobalOptions(doc);
        
        List outList=new ArrayList();
        
        NodeList stagesNodeList=(NodeList)xpath.evaluate("/executions/execution",doc,XPathConstants.NODESET);
        
        if(stagesNodeList!=null){
        	int i;
        	for(i=0;i<stagesNodeList.getLength();i++){
        		Execution stage=parse(stagesNodeList.item(i));
        		ExecutionData stageData=stage.parseData(this,stagesNodeList.item(i));
        		stage.setData(stageData);
        		
        		executions.add(stage);
        		
                if(stage.getName()==null) throw new SAXException("execution/@name is mandatory");
        		executionMap.put(stage.getName(),stage);
        	}
        } 
        
        
        
	}

	private void parseGlobalOptions(Document doc) throws Exception{
		 XPathFactory factory = XPathFactory.newInstance();
	     XPath xpath = factory.newXPath();
	     NodeList globals= (NodeList)xpath.evaluate("./executions/global",doc,XPathConstants.NODESET);
	     for(int i=1;i<=globals.getLength();i++){
	    	 String name=(String)xpath.evaluate("./executions/global["+i+"]/@name",doc,XPathConstants.STRING);
			 String value=(String)xpath.evaluate("./executions/global["+i+"]/text()",doc,XPathConstants.STRING);
			//resolvemos las variables globales dinámicas a partir de las demás variables globales definidas anteriormente
			 globalVariables.put(name, DynamicQueryResolver.resolveString(value,globalVariables));
	     }
	     
	}

	public static Execution parse(Node item) throws Exception{
        // Inicializar los objetos para las busquedas de elementos en el xml
        XPathFactory factory = XPathFactory.newInstance();
        XPath xpath = factory.newXPath();
		
		String 	_name=				item.getAttributes().getNamedItem("name").getNodeValue();
		String _tipus=				(item.getAttributes().getNamedItem("type")!=null)?item.getAttributes().getNamedItem("type").getNodeValue():null;
		String _description = 		((String)xpath.evaluate("./description",item,XPathConstants.STRING));
		
		if(_tipus==null)throw new SAXException("atribute \"type\" not defined for node "+_name);
		

		
		//creem una instància de la classe difinida a tipus. Si tipus=null, per defecte agafem StageInfoDocumentManager
		Class stageTipusClass=Execution.class.getClassLoader().loadClass(_tipus);
		Constructor constructor=stageTipusClass.getDeclaredConstructor(new Class [] {String.class,String.class});
		Execution instance=(Execution)constructor.newInstance(new Object[] {_name,_description});
		
		return instance;
	}
	
	public String uploadExecutionResultFile(Object in, String executionName,String resultContentType) throws Exception{
		String uploadServlet=getMonitorServletURL().replace("ResultsCollectorServlet", "UploadServlet");
		String downloadServlet=getMonitorServletURL().replace("ResultsCollectorServlet", "DownloadServlet");
		URL uploadServletURL=new URL(uploadServlet);
		
		// Generar la firma
		HttpURLConnection c = (HttpURLConnection) uploadServletURL.openConnection();
		c.setRequestProperty("configFilePath", getConfigURL());
		c.setRequestProperty("executionName", executionName);
		c.setDoInput(true);
		c.setDoOutput(true);
		c.setRequestMethod("POST");
		c.setRequestProperty("content-type", resultContentType);
		c.connect();
		ObjectOutputStream o = new ObjectOutputStream(c.getOutputStream());
		
		o.writeObject(in);
		
		o.close();
		String response=readInputStream(c.getInputStream());
		c.disconnect();
		return downloadServlet+"/"+response;
	}

	
	private int executeRemoteVerification(Execution ex) throws Exception{
		String remoteVerificationServlet=getMonitorServletURL().replace("ResultsCollectorServlet", "RemoteVerificationServlet");
		URL uploadServletURL=new URL(remoteVerificationServlet);
		RemoteExecutionVerificationRequest verificationReq=new RemoteExecutionVerificationRequest(ex, getGlobalVariables());
		
		// Generar la firma
		HttpURLConnection c = (HttpURLConnection) uploadServletURL.openConnection();
		c.setDoInput(true);
		c.setDoOutput(true);
		c.setRequestMethod("POST");
		c.connect();
		ObjectOutputStream o = new ObjectOutputStream(c.getOutputStream());
		o.writeObject(verificationReq);
		o.flush();
		o.close();
		String response=readInputStream(c.getInputStream());
		c.disconnect();
		if(c.getResponseCode()!=200)
			return Execution.VERIFICATION_KO;
		else
			return Integer.parseInt(response);
	}
	
	private String readInputStream(InputStream inputStream) throws IOException {
		ByteArrayOutputStream out=new ByteArrayOutputStream();
		BufferedInputStream in = new BufferedInputStream (inputStream);
        byte [] b = new byte [8192];
        
        int read=0;
        while ((read = in.read(b)) > 0)
        {
        	out.write(b,0,read);
        }
        in.close();
        
        return out.toString("UTF-8");

		
	}
	
	/**
	 * @return the view
	 */
	public synchronized ExecutionView getView() {
		return view;
	}

	/**
	 * @param view the view to set
	 */
	public synchronized void setView(ExecutionView view) {
		this.view = view;
	}

	/**
	 * @return the config
	 */
	public synchronized String getConfig() {
		return config;
	}

	/**
	 * @param config the config to set
	 */
	public synchronized void setConfigURL(String configFileURL) {
		this.configURL=configFileURL;
		
		String conf=null;
		try{
			URL f=new URL(configFileURL);
			InputStream in=f.openStream();
			ByteArrayOutputStream out=new ByteArrayOutputStream();
			byte buf[]=new byte[4096];
			int r=0;
			while((r=in.read(buf))!=-1){
				out.write(buf, 0, r);
			}
			conf=new String(out.toByteArray(),"UTF-8");
			String encodingtag=conf.substring(conf.indexOf("encoding"),conf.indexOf("?>"));
			String encoding=encodingtag.substring(encodingtag.indexOf("\"")+1, encodingtag.lastIndexOf("\""));
			conf=new String(out.toByteArray(),encoding);
		}catch(Exception e){
			view.notifyFatal("Error! "+e.getMessage(), e);
			return;
		}
		this.config = conf;
	}

	/**
	 * @return the executions
	 */
	public synchronized Vector getExecutions() {
		return executions;
	}

	/**
	 * @return the executionMap
	 */
	public synchronized Map getExecutionMap() {
		return executionMap;
	}

	/**
	 * Devuelve las variables globales, más las ejecuciones
	 * @return the globalVariables
	 */
	public synchronized Map getGlobalVariables() {
		Map o=new Hashtable();
		o.putAll(globalVariables);
		
		return o;
	}

	/**
	 * @param globalVariables the globalVariables to set
	 */
	public synchronized void setGlobalVariables(Map globalVariables) {
		this.globalVariables = globalVariables;
	}
	
	public synchronized String getMonitorServletURL() {
		return monitorServletURL;
	}
	
	public synchronized void setMonitorServletURL(String monitorServletURL) {
		this.monitorServletURL = monitorServletURL;
	}
	
	
	public synchronized String getConfigURL() {
		return configURL;
	}
}
