package es.caib.signatura.provider.impl.tradise;

import java.io.ByteArrayInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.StringReader;

import java.security.AccessController;
import java.security.PrivilegedActionException;
import java.security.PrivilegedExceptionAction;
import java.security.UnrecoverableKeyException;
import java.util.Date;
import java.util.Properties;
import java.util.Random;
import java.util.Vector;

import java.security.cert.CertificateException;
import java.security.cert.X509Certificate;

import org.bouncycastle.util.encoders.Base64;
 
import com.tradise.crypto.CryptoServiceProvider;
import com.tradise.crypto.certificate.CertificateStoreService;
import com.tradise.crypto.certificate.data.PrivateKeyCertificateChain;
import com.tradise.crypto.exception.AuthorizationException;
import com.tradise.crypto.exception.CommunicationException;
import com.tradise.crypto.exception.ServiceNotAvailableException;
import com.tradise.crypto.signature.SignatureService;
import com.tradise.crypto.signature.data.SignedData;
import com.tradise.crypto.signature.timestamp.TimestampService;
import com.tradise.crypto.signature.timestamp.data.TimestampResponder;

import es.caib.signatura.api.Signature;
import es.caib.signatura.api.SignatureCertNotFoundException;
import es.caib.signatura.api.SignatureDataException;
import es.caib.signatura.api.SignatureException;
import es.caib.signatura.api.SignaturePrivKeyException;
import es.caib.signatura.api.SignatureProviderException;
import es.caib.signatura.api.SignatureSignException;
import es.caib.signatura.api.SignatureTimestampException;
import es.caib.signatura.impl.SigDebug;
import es.caib.signatura.impl.ValidadorProxy;
import es.caib.signatura.impl.MIMEInputStream;
import es.caib.signatura.impl.SignerProviderInterface;
import es.caib.signatura.provider.impl.common.PDFSigner;
import es.caib.signatura.provider.impl.common.ParsedCertificateImpl;
import es.caib.signatura.provider.tradise.TradiseSignature;
import es.caib.signatura.provider.tradise.TradiseSignatureRaw;
import es.caib.signatura.provider.tradise.TradiseSignatureTest;
import es.caib.signatura.provider.tradise.TradiseSignatureRawTest;



/**Implementa la interfaz <code>Signature</code> para el proveedor de firma electrónica TRADISE
 * Tradise implementa la firma electrónica con y sin sello de tiempo
 * @author 3digits
 * @version 1.0
 * @see Signature
 *
 */
public class TradiseSigner implements SignerProviderInterface {
	private static final String TIMESTAMP_SERVICE = "Timestamp Service";
	protected static final String RECOGNIZED_CERTIFICATE_OID ="1.3.6.1.4.1.18332.40";
	protected static final String SIGNATURE_BUILDER = "CAIB";
	protected static final String SIGNATURE_APP = "SIGNATURE-API";
	protected static final String SIGNATURE_FORMAT = "CMS";
	protected static final String EXPECTED_API_VERSION = "2.5";
	private String storeFile = null;
	private String storePassword = null;
	boolean production;
	
	public TradiseSigner() {
		try{
			storeFile = System.getProperty("tradise.store.file");
			if( SigDebug.isActive() )  SigDebug.write("TradiseSigner: El repositori del servidor es: "+storeFile);
		    storePassword = System.getProperty("tradise.store.password");
		     // Verificar la versión del API
		    //TODO: general exception
		} catch (Exception e) {
			
		}
		
	    InputStream inputStream =  getClass().getClassLoader().getResourceAsStream("Service.properties");
	    if (inputStream==null) 
	    {
	    	throw new RuntimeException("No te instal·lat l'API de signatura");
	    }
	    Properties tempProperties = new Properties();
	    try {
			tempProperties.load(inputStream);
		    inputStream.close();
		} catch (IOException e) {
			e.printStackTrace();
		}
	    String version = (String) tempProperties.get("TradiseApiVersion");
	    if( SigDebug.isActive() )  SigDebug.write("TradiseSigner: Versió de l'API de tradise instal.lada al client: "+version);
	    if (version == null)
	    {
	    	throw new RuntimeException("No te instal·lada la versió "+EXPECTED_API_VERSION+" o superior de l'API de signatura");
	    }
	    String split [] = version.split("\\.");
	    String split2 [] = EXPECTED_API_VERSION.split("\\.");
	    for (int i = 0; i < split2.length; i ++)
	    {
	    	if ( i >= split.length)
		    	throw new RuntimeException("No te instal·lada la versió "+EXPECTED_API_VERSION+" o superior de l'API de signatura");
	    	int v1 = Integer.parseInt(split[i]);
	    	int v2 = Integer.parseInt(split[i]);
	    	if (v1 > v2){ // Versión superior
	    		if( SigDebug.isActive() )  SigDebug.write("TradiseSigner: Versió superior.");
	    		break;
	    	}
	    	if (v1 < v2) { // Versión inferior
	    		if( SigDebug.isActive() )  SigDebug.write("TradiseSigner: Versió inferior."+ v1 + " < "+v2);
//		    	throw new RuntimeException("No te instal·lada la versió "+EXPECTED_API_VERSION+" o superior de l'API de signatura");
	    	}
	    }
	    
	    production = tempProperties.getProperty("TradiseTest") == null;
	    
	}

	public String[] getCertList(boolean recognized) throws  SignatureCertNotFoundException, SignaturePrivKeyException{
		
		//boolean valid = false;

	    if( SigDebug.isActive() )  SigDebug.write("TradiseSigner: getCertList entrat");

		CertificateStoreService certStoreService;
		try {
			certStoreService = getCertificateStore();
		} catch (SignatureCertNotFoundException e1) {
			throw new SignatureCertNotFoundException (e1);
		} catch (UnrecoverableKeyException e1) {
			throw new SignatureCertNotFoundException (e1);
		} catch (IOException e1) {
			throw new SignatureCertNotFoundException (e1);
		}
		//Obtenemos la lista de certificados con clave privada disponibles.
		PrivateKeyCertificateChain[] privateCertChains=null;	
		privateCertChains = certStoreService.getPrivateKeyCertificateChains();
		
		Vector vCertList = new Vector(privateCertChains.length);

		ValidadorProxy validador = new ValidadorProxy();
		
		for (int i = 0; i < privateCertChains.length; i++) {
			//valid=true;
			PrivateKeyCertificateChain privCertChain = privateCertChains[i];						
			//Dependiendo del tipo de certificado a obtener, almacenamos en la lista de certificados
			//aquéllos que sean de tipo reconocido (CR) o de tipo no reconocido (CNR)
			//Un certificado reconocido es aquél que tiene la extensión definida en la variable RECONIZED_CERTIFICATE_OID, definida por TRADISE
//			String extensionOID = CertificateUtil.getExtensionForOIDAsString(privCertChain.getMainCertificate(),TradiseSigner.RECOGNIZED_CERTIFICATE_OID);
	//		if ((( recognized && extensionOID!=null) || 
//				(! recognized && extensionOID==null)) && (valid))
			ParsedCertificateImpl parser;
			try {
				privCertChain.getMainCertificate().checkValidity();
				// Se suposa que els certificats en PKCS11 es troben en un device segur.
				parser = new ParsedCertificateImpl (privCertChain.getMainCertificateChain(), true);
			    if( SigDebug.isActive() )  SigDebug.write("TradiseSigner: getCertList certificat trobat " + parser.getCommonName() );			    
				if (  ( (production && ! parser.isTest()) ||
						(!production && parser.isTest()) ) &&
					( recognized && parser.isRecognized () ||
					  ! recognized && parser.isAdvanced() )
					)
				{
				    if(validador.isEnDesenvolupament()){
						vCertList.addElement(parser.getCommonName());
					    if( SigDebug.isActive() )  SigDebug.write("TradiseSigner: getCertList certificat afegit " + parser.getCommonName() );				    	
				    }else{
				    	if(!parser.isTest()){
				    		vCertList.addElement(parser.getCommonName());
				    		if( SigDebug.isActive() )  SigDebug.write("TradiseSigner: getCertList certificat afegit " + parser.getCommonName() );
				    	}
				    }
				}
			} catch (CertificateException e) {
			} catch (IOException e) {
				e.printStackTrace();
			}
			
		}
	    if( SigDebug.isActive() )  SigDebug.write("TradiseSigner: getCertList retornant");
		return (String[]) vCertList.toArray(new String[vCertList.size()]);	
	}

//	private String getCertificateName(PrivateKeyCertificateChain privCertChain) {
//		String name = CertificateUtil.getSubjectCommonName(privCertChain.getMainCertificate());
//		if ( ! production )
//			name = "TEST: "+name;
//		return name;
//	}
	
	
	private class GetCertificateStoreAction implements PrivilegedExceptionAction
	{
		public Object run() throws Exception {
			CertificateStoreService certStoreService ; 
			if ( storeFile != null && storePassword != null)
			{
				certStoreService = CryptoServiceProvider.getCertificateStoreService(storePassword, storeFile);
			}
			else
			{
				CertificateStoreService[] certStores = detectCertificateStores();
				if (certStores==null || certStores.length == 0)
				{
					if( SigDebug.isActive() )  SigDebug.write( "TradiseSigner: Error, cerStores="+certStores+" i longitud="+ certStores.length);
					throw  new SignatureCertNotFoundException();
				}	
				//Nos quedamos con el primer de los almacenes, en realidad, nos quedamos con el servicio que lo representa
				certStoreService = certStores[0];
			}
			return certStoreService;
		}
	}
	
	/**
	 * @return
	 * @throws SignatureCertNotFoundException
	 * @throws IOException
	 * @throws UnrecoverableKeyException
	 */
	private CertificateStoreService getCertificateStore() 
		throws SignatureCertNotFoundException, UnrecoverableKeyException, IOException {
		try {
			return (CertificateStoreService) AccessController.doPrivileged(new GetCertificateStoreAction());
		} catch (PrivilegedActionException e) {
			Exception e2 = e.getException();
			if (e2 instanceof SignatureCertNotFoundException)
				throw (SignatureCertNotFoundException) e2;
			if (e2 instanceof UnrecoverableKeyException)
				throw (UnrecoverableKeyException) e2;
			if (e2 instanceof IOException)
				throw (IOException) e2;
			if (e2 instanceof RuntimeException)
				throw (RuntimeException) e2;
			throw new RuntimeException (e);
		}
	}

	/**
	 * @return
	 * @throws SignatureCertNotFoundException
	 */
	private CertificateStoreService[] detectCertificateStores() throws SignatureCertNotFoundException {
		//		Detectamos los almacenes de certificados disponibles en el sistema
		CertificateStoreService[] certStores = null;
		try{
			certStores = CryptoServiceProvider.detectCertificateStores();
		}
		catch (java.lang.ExceptionInInitializerError initError)
		{
			throw new SignatureCertNotFoundException();
		}
		return certStores;
	}

	public Signature sign(InputStream contentStream, String certificateName, String password, String contentType, boolean recognized,
			boolean timestamp, boolean signRaw ) throws IOException, SignatureException {
		if ( SigDebug.isVerbose() ) 
			SigDebug.write("TradiseSigner: sign cert " + certificateName + ", recognized " + (recognized ? "Si" : "No") + ", timestamp "
					+ (timestamp ? "Si" : "No"));

		Signature signature = null;

		//Comprobamos que el certificado es del tipo adecuado 
		PrivateKeyCertificateChain privKeyCertChain = getPrivateKeyCertificateChain(certificateName,recognized);
		
	    if( SigDebug.isVerbose() ) SigDebug.write("TradiseSigner: sign firmant" );
		if (timestamp) // Firma Avanzada con sello de tiempo
		{
			try{
				signature = signTimestamped(contentStream,privKeyCertChain,password,contentType,signRaw);
			}catch(Exception e){
				throw new SignatureException(e.getMessage(), e.getCause());
				/*
				 * EN TRADISE NO -> Si hay algún problema con la firma con time stamp entonces se genera una
				 * firma sin timestamp con content type de time stamp (es inconsistente,
				 * el verificador da error).
				 */
				//signature = signAdvanced(mimeStream,privKeyCertChain,password,contentType);
			}
		} else {
			signature = signAdvanced(contentStream,privKeyCertChain,password,contentType,signRaw);
		}
	    if( SigDebug.isVerbose() ) SigDebug.write("TradiseSigner: sign retornant" );
		return signature;
	}
	
	public void signPDF(InputStream contentStream, OutputStream signedStream, String certificateName,
			String password, String contentType, boolean recognized, String url, int position) throws IOException,
			SignatureException {
		if ( SigDebug.isVerbose() ) 
			SigDebug.write("TradiseSigner: sign cert " + certificateName + ", recognized " + (recognized ? "Si" : "No"));

		//Comprobamos que el certificado es del tipo adecuado 
		PrivateKeyCertificateChain privKeyCertChain = getPrivateKeyCertificateChain(certificateName,recognized);
		
		try {
			PDFSigner.sign(contentStream, signedStream, privKeyCertChain.getPrivateKey(password), privKeyCertChain.getMainCertificateChain(), url, position);
		} catch (UnrecoverableKeyException e) {
			throw new SignatureException(e.getMessage(), e.getCause());
		} catch (Exception e) {
			throw new SignatureException(e.getMessage(), e.getCause());
		}

	}
	
	
	
	private  Signature signAdvanced (InputStream contentStream, PrivateKeyCertificateChain privKeyCertChain, String password, String contentType, boolean signRaw) 
		throws IOException, SignatureDataException, SignatureProviderException, SignatureSignException
	{
		Signature signatureData = null;
		
	    if( SigDebug.isVerbose() ) SigDebug.write("TradiseSigner: signAdvanced entrant" );

		SignatureService signatureService;
		try {
			signatureService = CryptoServiceProvider.getSignatureService(SIGNATURE_BUILDER, SIGNATURE_APP);
		} catch (ServiceNotAvailableException e) {
			throw new SignatureProviderException(e);
		}
		if (signatureService==null)
		{
			throw new SignatureProviderException();
		}	
		
		SignedData signedData = null;
		if( signRaw ) {
			signedData = signatureService.initSignedData(contentStream, null, SIGNATURE_FORMAT);
		}
		else
		{
			InputStream mime = new MIMEInputStream( contentStream, contentType );
		
			signedData = signatureService.initSignedData(mime, null, SIGNATURE_FORMAT);
		}

		try {
			//Añadimos una firma electrónica
			//Si la contraseña no es correcta se lancará una excepción UnrecoverableKeyException;
			signedData.addSimpleSignature(privKeyCertChain, password.toCharArray());
		} catch (UnrecoverableKeyException e1) {
		    if( SigDebug.isVerbose() ) SigDebug.write("TradiseSigner: signAdvanced UnrecoverableKeyException" );
			throw new SignatureSignException ("Contrassenya incorrecta", e1);
		}
		
		signatureData = getSignatureData( production, signRaw, signedData, contentType );
		//contentStream.close();
		
	    if( SigDebug.isVerbose() ) SigDebug.write("TradiseSigner: signAdvanced retornant" );

	    byte [] arr= signatureData.getPkcs7();
	    	
	    return signatureData;
	}
	
	private Signature getSignatureData( boolean parmProduction, boolean signRaw, 
			SignedData signedData, String contentType )
		throws SignatureDataException
	{
		if (parmProduction)
		{
			if( signRaw )
			{
				return new TradiseSignatureRaw(
					new TradiseSignatureRawImpl (signedData, contentType));
			}
			else
			{
				return new TradiseSignature(
					new TradiseSignatureImpl (signedData, contentType));
			}
		}
		else
		{
			if( signRaw )
			{
				return new TradiseSignatureRawTest(
					new TradiseSignatureRawImpl (signedData, contentType));
			}
			else
			{
				return new TradiseSignatureTest(
					new TradiseSignatureImpl (signedData, contentType));
			}
		}
	}
	
	
	
	/** Obtiene la cadena de certificación privada a partir del nombre del certificado
	 *  @param certificateCommonName String nombre del certificado (campo CN de un certificado)
	 *  @return PrivateKeyCertificateChain
	 */
	private PrivateKeyCertificateChain getPrivateKeyCertificateChain(String certificateCommonName, boolean recognizedCertificate)
		throws SignatureProviderException, SignatureCertNotFoundException, SignaturePrivKeyException
	{
		
		CertificateStoreService certStoreService;
		try {
			certStoreService = getCertificateStore();
		} catch (UnrecoverableKeyException e) {
			throw new SignatureCertNotFoundException (e);
		} catch (IOException e) {
			throw new SignatureCertNotFoundException (e);
		}
		//Obtenemos la lista de certificados con clave privada disponibles.
		PrivateKeyCertificateChain[] privateCertChains = null;
		privateCertChains = certStoreService.getPrivateKeyCertificateChains();
		
		
		boolean certificateFound = false;
		PrivateKeyCertificateChain privCertChain = null;
		
		for (int i = 0; i < privateCertChains.length && !certificateFound; i++) 
		{
			privCertChain = privateCertChains[i];
			//Obtenemos el campo "COMMONNAME" del certificado para presentarlo por pantalla
			try {
				X509Certificate certs[] = privCertChain.getMainCertificateChain();
				certs[0].checkValidity();
				// Se suposa que els certificats en PKCS11 es troben en un device segur.
				ParsedCertificateImpl parser = new ParsedCertificateImpl (certs, true );
				//Comprobamos si hemos encontrado el certificado
				if (  parser.getCommonName().equals(certificateCommonName) &&
						(production && ! parser.isTest() ||
						!production && parser.isTest() ) &&
					( recognizedCertificate && parser.isRecognized () ||
					  ! recognizedCertificate && parser.isAdvanced() )
					)
				{
					certificateFound = true;
				}
			} catch (IOException e) {
			} catch (CertificateException e) {
			} 
		}
		
		if (!certificateFound) { 
			if( SigDebug.isActive() )  SigDebug.write("TradiseSigner: Certificat no trobat.");
			throw new SignatureCertNotFoundException();

		}
		else
			return privCertChain;
		
	}
	
	
	private  Signature signTimestamped (InputStream contentStream, PrivateKeyCertificateChain privKeyCertChain, String password, String contentType, boolean signRaw ) 
	throws IOException, SignatureDataException, SignatureSignException, SignatureTimestampException, SignatureProviderException
	{
	    if ( SigDebug.isVerbose() ) 
			SigDebug.write("TradiseSigner: signTimestamped entrant");

		Signature signatureData = null;
		SignatureService signatureService;
		TimestampService tsService;

		try {
			// Obtener el servicio de firma
			signatureService = CryptoServiceProvider.getSignatureService(SIGNATURE_BUILDER, SIGNATURE_APP);
			// Obtener el servicio de sello de tiempo
			tsService = getTimestampService(privKeyCertChain);
		} catch (ServiceNotAvailableException e3) {
			throw new SignatureProviderException(e3);
		}
		if (signatureService == null)
			throw new SignatureProviderException();

		SignedData signedData = null;
		// if(contentStream == null){
		// throw new SignatureDataException("The content is null.");
		// }

		if( signRaw ) {
			signedData = signatureService.initSignedData(contentStream, null, SIGNATURE_FORMAT);
		}
		else
		{
			signedData = signatureService.initSignedData(new MIMEInputStream( contentStream, contentType ), null, SIGNATURE_FORMAT);
		}

		// Añadimos una firma electrónica
		TimestampResponder tsResponders[];
		try {
			tsResponders = tsService.getResponders();
		} catch (CommunicationException e) {
			throw new SignatureTimestampException(e);
		}
		if(tsResponders.length == 0){
			throw new SignatureTimestampException("No responders found.");
		}
		SignatureTimestampException lastException = null;
		boolean ok = false;		
		int j = new Random().nextInt(tsResponders.length);
		for (int i = 0; !ok && tsResponders != null && i < tsResponders.length; i++) {
			try {
				int index = (i + j) % tsResponders.length;
				tsResponders[index].connect(privKeyCertChain, password.toCharArray());
				signedData.addLongTermSignature(privKeyCertChain, password.toCharArray(), tsResponders[index]);
				ok = true;
			} catch (UnrecoverableKeyException e1) {
				if ( SigDebug.isVerbose() ) 
					SigDebug.write("TradiseSigner: signTimestamped UnrecoverableKeyException");
				throw new SignatureSignException("TradiseSigner: signTimestamped UnrecoverableKeyException", e1);
				// TODO: excepción genérica
			} catch (Exception e) {
				lastException = new SignatureTimestampException(e);
			}

		}

		if (!ok) {
			if ( SigDebug.isVerbose() ) {
				SigDebug.write("TradiseSigner: signTimestamped ! ok");
			}
			if (lastException != null){
				// TODO: excepción genérica
				throw lastException;
			}else{
				throw new SignatureProviderException("No hi ha cap proveidor de segells de temps");
			}
		}
		// Si la firma no es correcto se lanza UnrecoverableKeyException

		// Si ha habido problemas de comunicación con el servidor de sello de
		// tiempo se lanza IOException
		signatureData = getSignatureData( production, signRaw, signedData, contentType );

		contentStream.close();

		if ( SigDebug.isVerbose() ) {
			SigDebug.write("TradiseSigner: signTimestamped retornant");
		}
		return signatureData;
	}
	
	
	public void generateSMIME (InputStream document, Signature signature, OutputStream smime) throws IOException
	{
		
		//String smimeText = new String("MIME-Version: 1.0\r\n");
		String smimeText = new String("");
		smimeText = smimeText + "Content-Type: multipart/signed; protocol=\"application/pkcs7-signature\"; micalg=\"sha1\"; ";
		smimeText = smimeText + "boundary=\"govern-de-les-illes-balears-boundary\"\r\n\r\n";
		smimeText = smimeText + "--govern-de-les-illes-balears-boundary\r\n";
		
		smime.write(smimeText.getBytes("UTF-8"));    
		//ByteArrayOutputStream mime = new ByteArrayOutputStream();
		
		//documentToMime(document,signature.getContentType(),mime);
		MIMEInputStream mime = new MIMEInputStream(document,signature.getContentType());
		int n;
		
		//Copiamos el mime en el smime
		byte[] byteBuffer = new byte[256];
		while ((n=mime.read(byteBuffer))!=-1)
		{
			smime.write(byteBuffer,0,n);
		}
		
		
		
		smimeText = "\r\n--govern-de-les-illes-balears-boundary\r\n";
		smimeText = smimeText + "Content-Type: application/pkcs7-signature; name=\"smime.p7s\" smime-type=signed-data\r\n";
		smimeText = smimeText + "Content-Transfer-Encoding: base64\r\n";
		smimeText = smimeText + "Content-Disposition: attachment; filename=\"smime.p7s\"\r\n";
		smimeText = smimeText + "Content-Description: S/MIME Cryptographic Signature\r\n\r\n";
		smime.write(smimeText.getBytes("UTF-8"));
		
		byte[] p7Base64 = Base64.encode(signature.getPkcs7());
		StringReader reader = new StringReader(new String(p7Base64,"UTF-8"));
		StringBuffer text = new StringBuffer();
		n=0;
		char[] buffer = new char[64];
		while ((n=reader.read(buffer,0,64))>-1)
		{
			text.append(buffer,0,n);
			text.append("\r\n");      
		} 
		
		text.append("\r\n--govern-de-les-illes-balears-boundary--\r\n");
		smime.write(text.toString().getBytes("UTF-8"));
		smime.flush();
		smime.close();
		document.close();
		
		
	}

	/* (non-Javadoc)
	 * @see es.caib.signatura.api.Signer#getCurrentDate()
	 */
	public Date getCurrentDate(String certificateName, String password,  boolean recognized) 
		throws SignatureException, IOException
	{
	    try {
			PrivateKeyCertificateChain privateKeyCertificateChain = 
				getPrivateKeyCertificateChain(certificateName, recognized);
		    //Obtenemos el servicio de sello de tiempo "Timestamp Service" asociado al certificado
			TimestampService tsService = getTimestampService(privateKeyCertificateChain); 
			//Obtenemos todos los notarios electrónicos asignados al servicio de sello de tiempo "Timestamp Service"
			TimestampResponder[] enotarios = tsService.getResponders();
			//Nos quedamos con el primer notario electrónico
			TimestampResponder enotary = enotarios[0];
			//Nos conectamos al notario (hay que tener contratado el servicio para que no salte una excepción, aunque
			//los certificados de test siempre tienen contratado este servicio)
			enotary.connect(privateKeyCertificateChain, password.toCharArray());
 
			//Finalmente obtenemos la hora del notario electrónico
			return new Date (enotary.getServerTime().getTime());
		} catch (ServiceNotAvailableException e) {
			throw new SignatureTimestampException (e);
		} catch (CommunicationException e) {
			throw new SignatureTimestampException (e);
		} catch (UnrecoverableKeyException e) {
			throw new SignatureTimestampException (e);
		} catch (AuthorizationException e) {
			throw new SignatureTimestampException (e);
		} 
	}

	private TimestampService getTimestampService ( PrivateKeyCertificateChain key) throws ServiceNotAvailableException, SignatureProviderException
	{	
		return CryptoServiceProvider.getTimestampService(SIGNATURE_BUILDER, TIMESTAMP_SERVICE);
	}

	public String getVersion() {
		return "3.0.0";
	}

}




