package es.caib.signatura.provider.impl.tradise;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.security.cert.CertificateExpiredException;
import java.security.cert.CertificateNotYetValidException;
import java.security.cert.X509Certificate;
import java.util.Collection;
import java.util.Date;
import java.util.Iterator;
import java.util.List;

import com.tradise.crypto.CryptoServiceProvider;
import com.tradise.crypto.certificate.util.CertificateUtil;
import com.tradise.crypto.exception.BadFormatException;
import com.tradise.crypto.exception.ServiceNotAvailableException;
import com.tradise.crypto.signature.SignatureService;
import com.tradise.crypto.signature.data.SignatureCertificateChain;
import com.tradise.crypto.signature.data.SignedData;
import com.tradise.crypto.signature.timestamp.data.TimestampTokenCertificateChain;

import es.caib.signatura.api.ParsedCertificate;
import es.caib.signatura.api.Signature;
import es.caib.signatura.api.SignatureDataException;
import es.caib.signatura.api.SignatureProviderException;
import es.caib.signatura.api.SignatureTimestampException;
import es.caib.signatura.api.SignatureVerifyException;
import es.caib.signatura.api.Signer;
import es.caib.signatura.impl.MIMEInputStream;
import es.caib.signatura.impl.ParsedCertificateProxy;
import es.caib.signatura.impl.SignaturaProperties;
import es.caib.signatura.impl.SignatureProviderInterface;
import es.caib.signatura.provider.impl.common.ParsedCertificateImpl;

/**
 * Implementación de la interfaz <code>SignatureProviderInterface</code> para usar con la
 * entidad certificadora Tradisea
 * 
 * @author 3digits
 * @version 0.98
 * @see Signer
 * @see Signature
 */
public class TradiseSignatureImpl implements SignatureProviderInterface {
	private SignedData signedData;

	private X509Certificate certificateChain [];

	private String contentType;

	private byte[] signatureBytes;
	
	protected static final String SIGNATURE_BUILDER = "CAIB";
	
	protected TradiseSignatureImpl(SignedData signedData, String contentType)
			throws SignatureDataException {
		ByteArrayOutputStream signatureByteArray = new ByteArrayOutputStream();
		try {
			getSignatureService()
					.saveSignatures(signedData, signatureByteArray);
		} catch (IOException e) {
			throw new SignatureDataException(e);
		}

		this.signedData = signedData;
		signatureBytes = signatureByteArray.toByteArray();
		this.contentType = contentType;
		extractCertificate();
	}

	public TradiseSignatureImpl ()
	{
	}
	
	public X509Certificate[] getCertificateChain() throws Exception{
		return certificateChain;
	}
	
	/**
	 * Crea un nuevo objeto a partir de los atributos de la clase. Es el
	 * constructor que debe usar cada implementación de la interfaz
	 * <code>Signature</code> para crear una firma. Se extrae el certificado
	 * de la firma y se guarda en la propiedad <code>transient</code>
	 * certificate para usarla en los métodos que dan información concreta del
	 * certificado
	 * 
	 * @param signatureBytes
	 *            array de bytes con la firma digital generada por la api del
	 *            proveedor de firma electrónica
	 * 
	 */
	
	public void setSignedData(byte[] pkcs7) throws Exception {
		try {
			signatureBytes = pkcs7;

			ByteArrayInputStream voidContentData = new ByteArrayInputStream(
					new byte[0]);
			signedData = getSignatureService().readSignedData(voidContentData,
					null, new ByteArrayInputStream(pkcs7),
					TradiseSigner.SIGNATURE_FORMAT);
			voidContentData.close();
		} catch (BadFormatException e) {
			throw new IOException("Unable to parse signed data");
		}
		extractCertificate();
	};

	/**
	 * Obtiene el nombre de la entidad certificadora usada en la firma
	 * 
	 * @return nombre de la entidad certificadora
	 */
	public String getCertCaName() {
		return certificateChain[certificateChain.length-1].getSubjectX500Principal().getName();
	}

	/**
	 * Obtiene el nombre del certificado usado en la firma
	 * 
	 * @return nombre del certificado (CommonName)
	 */

	public String getCertSubjectCommonName() {
		return CertificateUtil.getSubjectCommonName(certificateChain[0]);
	}

	public String getCertSubjectAlternativeNames() {

		StringBuffer altNameSB = new StringBuffer("");
		String altNameString = null;
		try {
			Collection generalNames = certificateChain[0].getSubjectAlternativeNames();
			Iterator itr = generalNames.iterator();
			while (itr.hasNext()) {
				List list = (List) itr.next();

				int tagNo = ((Integer) list.get(0)).intValue();
				switch (tagNo) {
				case 0:
					altNameSB.append("," + "otherName=");
					break;

				case 1:
					altNameSB.append("," + "rfc822Name=");
					break;

				case 2:
					altNameSB.append("," + "dNSName=");
					break;

				case 3:
					altNameSB.append("," + "x400Address=");
					break;

				case 4:
					altNameSB.append("," + "directoryName=");
					break;

				case 5:
					altNameSB.append("," + "ediPartyName=");
					break;

				case 6:
					altNameSB.append("," + "uniformResourceIdentifier=");
					break;

				case 7:
					altNameSB.append("," + "iPAddress=");
					break;

				case 8:
					altNameSB.append("," + "registeredID=");
					break;

				}
				altNameSB.append(list.get(1).toString());

				// Quitamos la coma del principio
				if (altNameSB.length() > 0) {
					altNameString = altNameSB.substring(1, altNameSB.length());
				}

			}
		}

		catch (Exception ex) {
			return null;
		}

		return altNameString;

	}

	private void extractCertificate() {
		SignatureCertificateChain chain[] = signedData.getSignatures();
		if (chain.length != 1)
			throw new RuntimeException("Unable to find signer");
		certificateChain = chain[0].getMainCertificateChain();
	}

	/**
	 * @param signatureService
	 * @return
	 * @throws SignatureProviderException
	 */
	private SignatureService getSignatureService() {
		try {
			// Obtener el servicio de firma
			return CryptoServiceProvider.getSignatureService(
					TradiseSigner.SIGNATURE_BUILDER,
					TradiseSigner.SIGNATURE_APP);
		} catch (ServiceNotAvailableException e3) {
			throw new RuntimeException(e3);
		}
	}

	public byte[] getPkcs7() {
		try {
			return this.signatureBytes;
		} catch (Exception ex) {

		}
		return null;
	}

	public Date getDate() throws SignatureTimestampException {
		SignatureCertificateChain chain[] = signedData.getSignatures();
		if (chain.length != 1)
			throw new RuntimeException("Unable to find signer");
		certificateChain = chain[0].getMainCertificateChain();

		Date timeStamp = null;
		// Utilizamos un array de bytes vacio como contenido firmado, la
		// verificación siempre devolverá incorrecto pero nos evitamos
		// excepciones
		// del tipo NullPointerException.
		SignatureService signatureService = getSignatureService();
		try {
			SignatureCertificateChain[] signatures = signedData.getSignatures(); 
			// Obtenemos las firmas (puede haber más de una)
			for (int i = 0; i < signatures.length; i++) {

				SignatureCertificateChain currentSignature = signatures[i];
				TimestampTokenCertificateChain tst[] = currentSignature
						.getTimestampTokenAttributes();
				// Comprobamos que la firma tenga sello de tiempo
				if (tst != null && tst.length > 0)
					timeStamp = tst[0].getTimestamp();
			}
		} catch (Exception ex) {
			throw new SignatureTimestampException(ex);
		}
		return timeStamp;

	}

	public boolean verify() {
		try {
			certificateChain[0].checkValidity();
		} catch (CertificateExpiredException cee) {
			System.out.println("TradiseSignature Certificat rebutjat, el certificat ha caducat.");
			return false;
		} catch (CertificateNotYetValidException cve) {
			System.out.println("TradiseSignature Certificat rebutjat, el certificat encara no és vàlid.");
			return false;
		}
		return true;
	}

	public X509Certificate getCert() {
		SignatureCertificateChain chain[] = signedData.getSignatures();
		if (chain.length != 1)
			throw new RuntimeException("Unable to find signer");
		return chain[0].getMainCertificate();
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see es.caib.signatura.api.Signature#verify(java.io.InputStream)
	 */
	public boolean verifyAPosterioriTimestamp(InputStream contentStream) 
		throws SignatureProviderException, IOException, SignatureVerifyException 
	{
		InputStream mimeContentStream = new MIMEInputStream(contentStream, getContentType());
		return verifyAPosterioriTimestampRaw( mimeContentStream );
	}

	protected boolean verifyAPosterioriTimestampRaw(InputStream contentStream) 
		throws SignatureProviderException, IOException, SignatureVerifyException 
	{
		//[-3d] contentStream = new MIMEInputStream(contentStream, getContentType());
		// Obtenemos el servicio de gestión de firmas
		boolean result = true;
		SignatureCertificateChain[] signaturesCertChain = null;

		SignatureService signatureService;
		try {
			signatureService = CryptoServiceProvider.getSignatureService(TradiseSigner.SIGNATURE_BUILDER, TradiseSigner.SIGNATURE_APP);
		} catch (ServiceNotAvailableException e) {
			throw new SignatureProviderException(e);
		}
		if (signatureService == null) {
			throw new SignatureProviderException();
		}

		ByteArrayInputStream signatureStream = null;

		if (signatureBytes == null) {
			return false;
		}

		signatureStream = new ByteArrayInputStream(signatureBytes);

		try {
			SignedData localSignedData = signatureService.readSignedData(contentStream, null, signatureStream, TradiseSigner.SIGNATURE_FORMAT);
			signaturesCertChain = localSignedData.getSignatures();
		} catch (com.tradise.crypto.exception.BadFormatException bfe) {
			throw new SignatureVerifyException(bfe.getCause());
		}

		// Comprobamos que la firma de todos los firmantes es válida. Si una es
		// inválida, el método
		// devuelve false

		for (int i = 0; i < signaturesCertChain.length; i++) {
			// Comprobamos la firma de uno de los firmantes
			result = result && signaturesCertChain[i].verify();
			if (result) {
				// comprobamos los timestamp de las firmas
				TimestampTokenCertificateChain timestampCertificateChain[] = signaturesCertChain[i].getTimestampTokenAttributes();
				if (timestampCertificateChain == null || timestampCertificateChain.length == 0) {
					/*FIXME Verify a posteriori no pone Timestamp y siempre devuelve false
					 *  Esto es lo que se debería hacer en caso de querer time stamp
					 * a posteriori: la firma no contiene un timestamp, hay que comprobar 
					 * que el timpo mime no tenga timestamp, en caso de serlo hay
					 * que intentar añadir el time stamp
					 */
					/* 
					 * la firma no contiene un timestamp, hay que comprobar que el timpo mime
					 * no tenga timestamp, en caso de serlo hay que devolver falso dado que la firma es
					 * inconsistente.
					 */
					SignaturaProperties signaturaProperties = new SignaturaProperties();
					if(signaturaProperties.needsTimeStamp(contentType)){
						return false;
					}
				} else {
					for (int j = 0; j < timestampCertificateChain.length; j++) {
						result = result && timestampCertificateChain[j].verify();
						if(j == 0){
							result = result && timestampCertificateChain[i].checkTimestampedData(signaturesCertChain[i].getSignature());
						}
					}
				}
			}
		}
		/* Se elimina la validación  por Web Services dado que el token asegura que
		 * el certificado no está revocado. Una vez Tradise tenga implementada en su
		 * totalidad el servidor OCSP entonces se puede plantear la posibilidad de 
		 * realizar una verificación adicional por Web Services
		try{
			String WSresponse = verifyWebServices(getCert().getEncoded(), getCertificateChain());		
			result = result && WSResponseIsAVerifiedMessage(WSresponse);
		}catch(Exception e){
			throw new SignatureVerifyException(e);
		}
		*/
		return result;
	}
	
	public boolean verify(InputStream contentStream)
		throws SignatureProviderException, IOException, SignatureVerifyException 
	{				
		InputStream mimeContentStream = new MIMEInputStream(contentStream, getContentType());
		
		/*FileOutputStream fo = new FileOutputStream("mime_verify.txt");
		
		while(true){
			int c=mimeContentStream.read();
			if(c!=-1)
				fo.write(c);
			else
				break;
		}*/
		
		return verifyRaw( mimeContentStream );
	}
	
	protected boolean verifyRaw(InputStream contentStream)
			throws SignatureProviderException, IOException,
			SignatureVerifyException {				

		// Obtenemos el servicio de gestión de firmas
		boolean result = true;
		SignatureCertificateChain[] signaturesCertChain = null;

		SignatureService signatureService;
		try {
			signatureService = CryptoServiceProvider.getSignatureService(
					TradiseSigner.SIGNATURE_BUILDER,
					TradiseSigner.SIGNATURE_APP);
		} catch (ServiceNotAvailableException e) {
			throw new SignatureProviderException(e);
		}
		if (signatureService == null) {
			throw new SignatureProviderException();
		}

		ByteArrayInputStream signatureStream = null;

		if (signatureBytes == null) {
			return false;
		}

		signatureStream = new ByteArrayInputStream(signatureBytes);

		try {
			SignedData localSignedData = signatureService.readSignedData(
					contentStream, null, signatureStream,
					TradiseSigner.SIGNATURE_FORMAT);			
			signaturesCertChain = localSignedData.getSignatures();
		} catch (com.tradise.crypto.exception.BadFormatException bfe) {
			throw new SignatureVerifyException(bfe.getCause());
		}

		// Comprobamos que la firma de todos los firmantes es válida. Si una es
		// inválida, el método
		// devuelve false
		
		for (int i = 0; i < signaturesCertChain.length; i++) {
			// Comprobamos la firma de uno de los firmantes
			result = result && signaturesCertChain[i].verify();
			
			if(result){
				//comprobamos los timestamp de las firmas
				TimestampTokenCertificateChain timestampCertificateChain[] 
				                                         				= signaturesCertChain[i].getTimestampTokenAttributes();
				if(timestampCertificateChain == null || timestampCertificateChain.length == 0){
					/* 
					 * la firma no contiene un timestamp, hay que comprobar que el timpo mime
					 * no tenga timestamp, en caso de serlo hay que devolver falso dado que la firma es
					 * inconsistente.
					 */
					SignaturaProperties signaturaProperties = new SignaturaProperties();
					if(signaturaProperties.needsTimeStamp(contentType)){
						return false;
					}
				}else{
					for(int j = 0;j < timestampCertificateChain.length; j++){
						result = result && timestampCertificateChain[j].verify();
						if(j == 0){
							result = result && timestampCertificateChain[i].checkTimestampedData(signaturesCertChain[i].getSignature());
						}
					}
				}
			}
		}
		/*	
		 * try{
		 * 		String WSresponse = verifyWebServices(getCert().getEncoded(), getCertificateChain());
		 * 		result = result && WSResponseIsAVerifiedMessage(WSresponse);
		 * }catch(Exception e){
		 * 		throw new SignatureVerifyException(e);
		 * }
		 */
		return result;
	}

	public void setContentType(String contentType) throws Exception {
		this.contentType = contentType;
	}

	public String getContentType() {
		return contentType;
	}

	public ParsedCertificate getParsedCertificate() {
		try {
			//TODO posar-lo en cache
			// Se suposa que els certificats en tradise es troben en un device segur.
			return new ParsedCertificateProxy( new ParsedCertificateImpl(certificateChain, false));
		} catch (Exception e) {
			throw new RuntimeException(e);
		}
	}
}